/*
 * Created on May 23, 2005
 * 
 * $Id: EmployersReferenceRule.java,v 1.3 2006/08/10 15:19:16 dec Exp $
 */
package paye.eoy.bizrules.header;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import paye.multiyear.TaxYear;
import xmldoc.Attribute;
import xmldoc.DocumentError;
import bizrules.AbstractBusinessRule;

/**
 * @author Rob
 * @since 23 May 2005
 */
public class EmployersReferenceRule extends AbstractBusinessRule {

    private static final TaxYear[] APPLICABLE_TAX_YEARS = TaxYear.allActive();

    protected static final Pattern CHARACTER_SET_B = Pattern.compile("[a-zA-Z0-9.()/&*'-]*");

    public static final String STARTS_WITH_SPACE = "2";

    public static final String TOO_LONG = "1";

    public static final String MANDATORY = "mandatory";

    private String taxOfficeReference = "";

    private boolean isTaxOfficeReference;

    private boolean taxOfficeReferenceSet = false;

    private final String[] subcodes = { STARTS_WITH_SPACE, TOO_LONG, MANDATORY, DocumentError.FORMAT };

    public EmployersReferenceRule() {
        getErrorBinder().add("/IRenvelope/IRheader/Keys/Key", "handleError");
        getValueBinder().add("/IRenvelope/IRheader/Keys/Key", "key");
    }

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public void setKey(final String value) {
        if (isTaxOfficeReference) {
            taxOfficeReference = value;
            taxOfficeReferenceSet = true;
        }
    }

    public void setKey(final Attribute[] attributes) {
        if ("".equals(taxOfficeReference)) {
            for (int i = 0; i < attributes.length && !isTaxOfficeReference; i++) {
                if ("Type".equals(attributes[i].getLocalName())) {
                    isTaxOfficeReference = "TaxOfficeReference".equals(attributes[i].getValue());
                    break;
                }
            }
        }
    }

    public void processRule() {
        if (taxOfficeReferenceSet) {
            if (taxOfficeReference.length() > 10) {
                raiseError(TOO_LONG);
            } else if (taxOfficeReference.length() == 0) {
                raiseError(DocumentError.BLANK);
            } else if (taxOfficeReference.startsWith(" ")) {
                raiseError(STARTS_WITH_SPACE);
            } else {
                checkCharacterSetB(taxOfficeReference);
            }
        } else {
            raiseError(MANDATORY);
        }
    }

    @Override
    public boolean handleError(DocumentError documentError) {
        /*
         * processRule() applies a character set b test to the Tax Office
         * Reference, which is more stringent than the IRheaderStringType
         * defined for Key in the schema, so suppress the schema error and let
         * the rule deal with bad format.
         */
        if (isTaxOfficeReference)
            return true;
        else
            return super.handleError(documentError);
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/IRheader" };
    }

    @Override
    public boolean isActive() {
        return true;
    }

    @Override
    protected String[] getApplicableNamespaces() {
        String[] namespaces = new String[APPLICABLE_TAX_YEARS.length];
        for (int i = 0; i < APPLICABLE_TAX_YEARS.length; i++) {
            namespaces[i] = APPLICABLE_TAX_YEARS[i].getIREnvelopeNamespaceString();
        }
        return namespaces;
    }

    @Override
    protected void initializeValueBindings() {
    }

    @Override
    protected void initializeEventBindings() {
    }

    protected void checkCharacterSetB(final String text) {
        final Matcher matcher = CHARACTER_SET_B.matcher(text);
        if (!matcher.matches()) {
            raiseError(DocumentError.FORMAT, text);
        }
    }
}
