/*
 * Created on 20-Jan-2006
 *
 * $Id$
 */
package paye.eoy.bizrules.p14;

import paye.eoy.bizrules.AbstractPAYEBusinessRule;
import paye.eoy.types.SubmissionType;
import xmldoc.DocumentEventType;
import bizrules.BusinessRuleError;
import bizrules.binding.EventBinder;
import bizrules.registry.DocumentEventDetail;

/**
 * All rules which deal with P14s should subclass this class which provides
 * access to the P14 locator details (surname, date of birth and nino). It also
 * triggers a call to the reset() method at the start of each P14 so that each
 * rule can reset itself to a clean state in preparation for processing the next
 * P14.
 * 
 */
public abstract class AbstractP14BusinessRule extends AbstractPAYEBusinessRule {

    public AbstractP14BusinessRule() {
        super();
        reset();

        /**
         * bind to the start of each P14 so we can make a new bean for the
         * personal details.
         */
        final EventBinder eventBinder = getEventBinder();
        eventBinder.add("/IRenvelope/EndOfYearReturn/P14Amended", "startP14", DocumentEventType.startElement);
        eventBinder.add("/IRenvelope/EndOfYearReturn/P14", "startP14", DocumentEventType.startElement);
    }

    /**
     * Resets any information that is p14 related. This is called at the start
     * of each P14 ( or P14 Amended )
     */
    protected abstract void reset();

    public final void startP14(final DocumentEventDetail detail) {
        reset();
    }

    protected final String getDateOfBirth() {
        return getP14Identifier().getDateOfBirth();
    }

    public final String getNino() {
        return getP14Identifier().getNino();
    }

    public final boolean isNinoPresent() {
        return getP14Identifier().isNinoPresent();
    }

    protected final String getSurname() {
        return getP14Identifier().getSurname();
    }

    public final void setDateOfBirth(final String value) {
        getP14Identifier().setDateOfBirth(value);
    }

    public final void setNino(final String value) {
        getP14Identifier().setNino(value);
    }

    protected P14Identifier getP14Identifier() {
        return P14IdentifierCollector.getP14Identifier();
    }

    @Override
    protected final BusinessRuleError raiseError(final String subcode, final Object valueInError) {
        return super.raiseError(subcode, valueInError, getP14Identifier());
    }

    @Override
    protected final BusinessRuleError raiseError(final String subcode) {
        return raiseError(subcode, "");
    }

    /**
     * Change the default from the AbstractPAYEBusinessRule class. We know that
     * P14 rules can never apply to P35Part submissions so remove that from the
     * list. Specific rules may restrict this further if they wish.
     */
    @Override
    protected final SubmissionType[] getActiveSubmissionTypes() {
        return new SubmissionType[] { SubmissionType.P14PART, SubmissionType.COMPLETE };
    }

}
