/*
 * Created on May 24, 2005
 *
 * $Id: EmployeeForenameRule.java,v 1.8 2006/08/10 15:19:16 dec Exp $
 */
package paye.eoy.bizrules.p14;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import paye.channel.SubmissionChannel;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;

/**
 * We need a rule to handle forenames because the rules for dealing with the
 * first forename entry are different from the rules for the second and
 * subsequent forenames.
 * 
 * @author Rob/Doug
 */
public final class EmployeeForenameRule extends AbstractP14BusinessRule {

    private final String[] subcodes = { BLANK, BLANK2, FORMAT, FORMAT2, MANDATORY };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    private List<String> forenames;

    private static final Pattern PATTERN = Pattern.compile("[A-Za-z][A-Za-z\'\\-]*");

    // Paper submissions do not have restriction on first char
    private static final Pattern PAPER_PATTERN = Pattern.compile("[A-Za-z\'\\-]*");

    public static final String BLANK = "blank";

    public static final String FORMAT = "format";

    public static final String BLANK2 = "blank2";

    public static final String FORMAT2 = "format2";

    public static final String MANDATORY = "mandatory";

    public void setForename(final String value) {
        forenames.add(value);
    }

    /**
     * Resets any information that is P14 related. This is called at the start
     * of each P14.
     */
    @Override
    protected void reset() {
        forenames = new ArrayList<String>();
    }

    public void processRule() {
        if (forenames.size() >= 1) {
            /*
             * We need to distinguish between the first forename and the second
             * forename. To do that we'll append a string to the subcode so that
             * we can set up the mappings accordingly. By default this is blank
             * so that the first forename will have conventional subcodes.
             */
            String subCodeSuffix = "";

            for (int i = 0; i < forenames.size(); i++) {
                final String forename = forenames.get(i);
                if ("".equals(forename)) {
                    raiseError(BLANK + subCodeSuffix, forename);
                } else {
                    if (forename.length() > 35 || !schemaCheck(forename)) {
                        raiseError(FORMAT + subCodeSuffix, forename);
                    }
                }
                subCodeSuffix = "2";
            }
        } else {
            // Note: only the first forename is mandatory (except on paper
            // submissions, where it is optional)
            if (getSubmissionChannel() != SubmissionChannel.Paper) {
                raiseError(MANDATORY);
            }
        }
    }

    private boolean schemaCheck(final String forename) {
        final Matcher matcher = getPattern().matcher(forename);
        final boolean matches = matcher.matches();
        return matches;
    }

    private Pattern getPattern() {
        if (getSubmissionChannel() == SubmissionChannel.Paper) {
            return PAPER_PATTERN;
        } else {
            return PATTERN;
        }
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected void initializeValueBindings() {
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/Name/Fore", "forename");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/Name/Fore", "forename");
    }

    /**
     * Because this rule is taking over from the schema check (see class javadoc
     * comment) we want to suppress any errors coming from the parser. For that
     * reason we bind this callback to errors on the Fore element and simply
     * return true to indicate that we have consumed them.
     * 
     * @return true in order to indicate we have consumed the error.
     */
    @Override
    public boolean handleError(final DocumentError error) {
        if (error.getSubCode().equals(BLANK) || error.getSubCode().equals(FORMAT) || error.getSubCode().equals(MANDATORY)) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P14Amended", "/IRenvelope/EndOfYearReturn/P14" };
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    @Override
    protected void initializeErrorBindings() {
        getErrorBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/Name/Fore", "handleError");
        getErrorBinder().add("/IRenvelope/EndOfYearReturn/P14/Name/Fore", "handleError");
    }

}
