/**
 * @author Rob
 */
package paye.eoy.bizrules.p14;

import paye.channel.SubmissionChannel;
import paye.eoy.types.ReturnType;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;

public final class EmployeeSurnameRule extends AbstractP14BusinessRule {
    public static final String FIRSTCHAR = "firstchar";

    public static final String MANDATORY = "mandatory";

    public static final String BLANK = "blank";

    public static final String FORMAT = "format";

    private String surname = null;

    private final String[] subcodes = { FIRSTCHAR, MANDATORY, BLANK, FORMAT };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public void processRule() {
        if (isNotBlank(surname)) {
            // first character must be a letter this is not being picked up
            // by the schema
            final char firstCharacter = getSurname().toUpperCase().charAt(0);

            final boolean firstCharacterIsAlpha = (firstCharacter >= 'A' && firstCharacter <= 'Z');

            if (!firstCharacterIsAlpha) {
                raiseError(FIRSTCHAR, getSurname());
            }
        } else {
            if (surname != null) {
                raiseError(BLANK);
            } else {
                // Surname is optional for Paper submissions.
                if (getSubmissionChannel() != SubmissionChannel.Paper) {
                    raiseError(MANDATORY);
                }
            }
        }
    }

    private boolean isNotBlank(final String string) {
        return string != null && !"".equals(string);
    }

    @Override
    protected ReturnType[] getActiveReturnTypes() {
        return new ReturnType[] { ReturnType.ORIGINAL, ReturnType.AMENDED };
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P14", "/IRenvelope/EndOfYearReturn/P14Amended" };
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    @Override
    protected void initializeValueBindings() {
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/Name/Sur", "surname");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/Name/Sur", "surname");
    }

    @Override
    protected void initializeErrorBindings() {
        getErrorBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/Name/Sur", "handleError");
        getErrorBinder().add("/IRenvelope/EndOfYearReturn/P14/Name/Sur", "handleError");
    }

    @Override
    public boolean handleError(final DocumentError error) {
        boolean result = false;

        if (error.getSubCode().equals(FORMAT)) {
            raiseError(FORMAT);
            result = true;
        } else if (error.getSubCode().equals(BLANK)) {
            // We're checking for blank in processRule so suppress the schema
            // error
            result = true;
        } else if (getSubmissionChannel() == SubmissionChannel.Paper) {
            final String subCode = error.getSubCode();
            if (subCode.equals(DocumentError.MANDATORY) || error.getSubCode().equals(DocumentError.INVALID)) {
                // Suppress mandatory check for Paper channel
                result = true;
            }
        }

        return result;
    }

    @Override
    protected void reset() {
        surname = null;
    }

    public void setSurname(final String surname) {
        this.surname = surname;
    }
}
