/*
 * Created on 07-Jan-2006
 *
 * $Id: AbstractAggregation.java,v 1.2 2006/06/08 13:49:07 dec Exp $
 */
package paye.eoy.bizrules.p14.aggregation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import paye.eoy.bizrules.helpers.BigDecimalUtil;
import paye.eoy.types.Nic;

/**
 * $Id: AbstractAggregation.java,v 1.2 2006/06/08 13:49:07 dec Exp $
 * 
 * This class iden
 */
abstract class AbstractAggregation implements Aggregation {

    /*
     * EARNING_COMBINATIONS contains an array of AggregationEarningsCombinations
     * objects that identify an aggregation case.
     * 
     * There are three earnings limits for each NIC tab element:
     * 
     * 1. AtLEL - Earnings at Lower Earnings Limit
     * 
     * 2. LELtoET - Earnings between Lower Earnings Limit and Earnings Threshold
     * 
     * 3. ETtoUEL - Earnings between Earnings Threshold and Upper Earnings
     * Limit.
     * 
     * The AggregationEarningsCombination object holds two strings of three Y's
     * and N's which correspond to whether the respective bands equal zero. The
     * first character corresponds to 1 above, and so forth ie
     * 
     * An 'N' determines an earnings limit != 0.
     * 
     * Conversely, 'Y' determines an earninge limt is zero.
     * 
     * There are two strings supplied in the AggregationEarningsCombination
     * constructor. The first parameter represents the first NIC (or "current"
     * NIC) AKA "Left" and the second parameter represents the second NIC (or
     * the "other" NIC), AKA "Right". Every current/left NIC Tab element is
     * cross-checked with all other/right NIC Tab elements to find out the
     * eligibility of an aggregation case based on the combinations of earnings
     * within each NIC Tab entry.
     */
    protected AggregationEarningsCombination[] getEarningsCombinations() {
        return new AggregationEarningsCombination[] { new AggregationEarningsCombination("NNN", "YYN"),
                new AggregationEarningsCombination("NNY", "YNN"), new AggregationEarningsCombination("NNY", "YNY"),
                new AggregationEarningsCombination("NYY", "YNN"),
                new AggregationEarningsCombination("NYY", "YNY"),
                new AggregationEarningsCombination("NNY", "YYN"),
                new AggregationEarningsCombination("NNN", "YNY"),
                new AggregationEarningsCombination("NNN", "YNN"),
                // the following is the inverse of the above
                new AggregationEarningsCombination("YYN", "NNN"), new AggregationEarningsCombination("YNN", "NNY"),
                new AggregationEarningsCombination("YNY", "NNY"), new AggregationEarningsCombination("YNN", "NYY"),
                new AggregationEarningsCombination("YNY", "NYY"), new AggregationEarningsCombination("YYN", "NNY"),
                new AggregationEarningsCombination("YNY", "NNN"), new AggregationEarningsCombination("YNN", "NNN") };
    }

    /**
     * Holds the aggregations categories.
     */
    protected static class AggregationNicCategoryCombination {

        public Character[] getRightCategories() {
            return rightCategories;
        }

        public Character getLeftCategory() {
            return leftCategory;
        }

        private final Character leftCategory;

        private final Character[] rightCategories;

        /**
         * @param leftCategory
         *            the category to check
         * @param rightString
         *            a string of many categories that make up an aggregation
         *            case if the left category also exists.
         */
        public AggregationNicCategoryCombination(final char leftCategory, final String rightString) {

            this.leftCategory = new Character(leftCategory);
            final char[] chars = rightString.toCharArray();

            rightCategories = new Character[rightString.length()];
            for (int i = 0; i < chars.length; i++) {
                rightCategories[i] = new Character(chars[i]);
            }
        }
    }

    /**
     * Holds the left and right states , each state is three character made up
     * from Y's and N's, for example NNN or NNY
     */
    protected static class AggregationEarningsCombination {

        private final String leftState;

        private final String rightState;

        public String getLeftState() {
            return leftState;
        }

        public String getRigthState() {
            return rightState;
        }

        public AggregationEarningsCombination(final String leftState, final String rightState) {
            this.leftState = leftState;
            this.rightState = rightState;
        }
    }

    private final Map<String, Collection<AggregationEarningsCombination>> EARNING_COMBINATIONS = toMapOfCollections(getEarningsCombinations());

    /**
     * @returns the state based on the nic, for example NNN or NNY
     */
    private String getState(final Nic nic) {

        class SateGenerator {

            private final StringBuffer state = new StringBuffer();

            public void add(final boolean value) {
                if (value) {
                    state.append("Y");
                } else {
                    state.append("N");
                }
            }

            @Override
            public String toString() {
                return state.toString();
            }
        }

        final SateGenerator sateGenerator = new SateGenerator();

        sateGenerator.add(nic.getContributionLineA().compareTo(BigDecimalUtil.ZERO) == 0);
        sateGenerator.add(nic.getContributionLineB().compareTo(BigDecimalUtil.ZERO) == 0);
        sateGenerator.add(nic.getContributionLineC().compareTo(BigDecimalUtil.ZERO) == 0);

        return sateGenerator.toString();
    }

    protected abstract Map<Character, AggregationNicCategoryCombination> getAggregationCases();

    /**
     * writes the StateTable[] to a Map of Collections keyed on the leftState
     */
    private static Map<String, Collection<AggregationEarningsCombination>> toMapOfCollections(
            final AggregationEarningsCombination[] stateTableArray) {

        final Map<String, Collection<AggregationEarningsCombination>> result = new HashMap<String, Collection<AggregationEarningsCombination>>();

        for (final AggregationEarningsCombination element : stateTableArray) {
            final String state = element.leftState;
            Collection<AggregationEarningsCombination> c = result.get(state);
            if (c == null) {
                result.put(state, c = new ArrayList<AggregationEarningsCombination>());
            }

            c.add(element);

        }

        return result;
    }

    /**
     * write the array into a hash map of collections for cross checking
     */
    private static Map<Character, Collection<Nic>> toMapOfCollections(final Nic[] nics) {
        final Map<Character, Collection<Nic>> result = new HashMap<Character, Collection<Nic>>();
        for (final Nic element : nics) {
            final Character niCategory = element.getNiCategory();
            Collection<Nic> c = result.get(niCategory);
            if (c == null) {
                result.put(niCategory, c = new ArrayList<Nic>());
            }

            c.add(element);

        }
        return result;
    }

    /**
     * writes the AggregationCase[] to a HashMap
     */
    protected static Map<Character, AggregationNicCategoryCombination> toMap(
            final AggregationNicCategoryCombination[] aggregationCase) {
        final Map<Character, AggregationNicCategoryCombination> result = new HashMap<Character, AggregationNicCategoryCombination>();
        for (final AggregationNicCategoryCombination element : aggregationCase) {
            result.put(element.leftCategory, element);
        }
        return result;
    }

    /**
     * Iterates through the supplied array of NICs and returns "true" if the
     * supplied combination of Nics is an aggregation case.
     * 
     * An aggregation case is determined by a specific (and somewhat
     * complicated) combination of NIC Tab entries and earning limits for each
     * category.
     * 
     * @return true if this is an aggregation case else false
     */
    public boolean isAggregation(final Nic[] nicArray) {

        final Map<Character, Collection<Nic>> nics = toMapOfCollections(nicArray);

        class CheckAggregation {

            /**
             * @returns true if the leftNic is a candidate for an aggregation
             *          case. Note that another entry in the nicArray would need
             *          to be a candidate for an aggregation case also in order
             *          for the P14 to be an aggregation case proper.
             */
            public boolean isAgregationCase(final Nic leftNic) {

                boolean result = false;

                // if nic category has a table letter that is eligible to become
                // an aggregation candidate, and the earnings limits are in a
                // specific state to also be considered as a potential
                // aggregation case, then we need to cross-check this entry with
                // all other/right NIC entries

                if (EARNING_COMBINATIONS.containsKey(getState(leftNic))
                        && getAggregationCases().containsKey(leftNic.getNiCategory())) {
                    result = isProvideNicsInAllowableRightStates(leftNic);
                }

                return result;
            }

            /**
             * Uses the left Nic entry to lookup the allowable states and
             * categories. This states and categories are then checked against
             * all the avalible nics in the P14.
             * 
             * @returns true
             */
            private boolean isProvideNicsInAllowableRightStates(final Nic leftNic) {

                final String leftState = getState(leftNic);
                final Map<Character, AggregationNicCategoryCombination> aggregationCases = getAggregationCases();

                // gets the allowable StateTable's based on the leftNic states
                final Collection<AggregationEarningsCombination> allowableStates = EARNING_COMBINATIONS.get(leftState);

                // gets the allowable AggregationCase's based on the leftNic
                // categories
                final AggregationNicCategoryCombination agregationCase = aggregationCases.get(leftNic.getNiCategory());

                // gets all the other/right categories based on the lefNic
                // categories from the aggregation case
                final Character[] allowableCategories = agregationCase.getRightCategories();

                boolean result = false;

                // Search and attempt to match combinations of NIC table
                // entries/allowable states of earnings limits.
                // returns true if a the allowable categories & states can be
                // found
                // in any of the available nics in the P14.
                for (int i = 0; i < allowableCategories.length && !result; i++) {

                    if (nics.containsKey(allowableCategories[i])) {
                        final Collection<Nic> rightNics = nics.get(allowableCategories[i]);

                        for (final Iterator<Nic> iter = rightNics.iterator(); iter.hasNext() && !result;) {
                            final Nic rightNic = iter.next();
                            {
                                final String rightState = getState(rightNic);
                                for (final Iterator<AggregationEarningsCombination> iterator = allowableStates.iterator(); iterator
                                        .hasNext()
                                        && !result;) {
                                    final AggregationEarningsCombination stateTable = iterator.next();
                                    result = stateTable.rightState.equals(rightState);
                                }
                            }

                        }
                    }
                }
                return result;
            }
        }

        final CheckAggregation checkAggregation = new CheckAggregation();

        boolean result = false;

        // itterates, cross checking each nic.
        for (int i = 0; i < nicArray.length && !result; i++) {
            result = checkAggregation.isAgregationCase(nicArray[i]);
        }

        return result;

    }
}
