package paye.eoy.bizrules.p14.br1112;

import java.math.BigDecimal;

import paye.eoy.bizrules.helpers.BigDecimalUtil;
import paye.eoy.bizrules.helpers.WholePoundsHelper;
import paye.eoy.bizrules.p14.AbstractP14BusinessRule;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.Attribute;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;

/**
 * The field ETtoUAP changed to PTtoUAP in 2011/12
 * 
 * @author Doug
 * @since 14/7/2010
 */
public final class EarningsPTtoUAPRule extends AbstractP14BusinessRule {

    public static final String EARNINGS_PT_TO_UAP_RULE = "EarningsPTtoUAPRule";

    public static final String MUST_BE_ZERO_FOR_NI_CATEGORY = "must be zero for ni category";

    private final String[] subcodes = { MUST_BE_ZERO_FOR_NI_CATEGORY, DocumentError.FORMAT, DocumentError.BLANK,
            DocumentError.MANDATORY, DocumentError.MINVALUE, DocumentError.MAXVALUE, WholePoundsHelper.NOT_WHOLE_POUNDS };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    private String niCategory = "";

    private BigDecimal earnings = BigDecimalUtil.ZERO;

    public EarningsPTtoUAPRule() {
        setRuleName(EARNINGS_PT_TO_UAP_RULE);
    }

    public void setNICs(final Attribute[] attribute) {

        for (final Attribute element : attribute) {
            if ("Tab".equals(element.getLocalName())) {
                final String value = element.getValue();

                setNiCategory(value);
            }
        }
    }

    @Override
    protected void initializeValueBindings() {

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/NICs", "NICs");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/NICs", "NICs");

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/NICs/Earn/PTtoUAP", "earnings");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/NICs/Earn/PTtoUAP", "earnings");

    }

    public void setEarnings(final String earnings) {
        this.setEarnings(BigDecimalUtil.toBigDecimal(earnings));
    }

    @Override
    protected void reset() {
        setNiCategory("");
        setEarnings(BigDecimalUtil.ZERO);
    }

    public void processRule() {
        checkSubrule4();
    }

    private void checkSubrule4() {
        if (getNiCategory().length() > 0 && getNICategoryForWhichValueMustBeZero().indexOf(getNiCategory()) != -1) {
            if (getEarnings().compareTo(BigDecimalUtil.ZERO) != 0) {
                raiseError(MUST_BE_ZERO_FOR_NI_CATEGORY, getEarnings());
            }
        }
    }

    protected String getNICategoryForWhichValueMustBeZero() {
        return "X";
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected final String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P14/NICs", "/IRenvelope/EndOfYearReturn/P14Amended/NICs" };
    }

    @Override
    protected final void initializeEventBindings() {
        // do nothing
    }

    private void setEarnings(final BigDecimal ettoUEL) {
        this.earnings = ettoUEL;
    }

    protected BigDecimal getEarnings() {
        return earnings;
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        final String[] errorHandlers = new String[] { "distinguishWholePoundsError", "handleError" };
        binder.add("/IRenvelope/EndOfYearReturn/P14/NICs/Earn/PTtoUAP", errorHandlers);
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/NICs/Earn/PTtoUAP", errorHandlers);
    }

    protected void setNiCategory(final String niCategory) {
        this.niCategory = niCategory;
    }

    protected String getNiCategory() {
        return niCategory;
    }
}
