package paye.eoy.bizrules.p14.nic;

import java.math.BigDecimal;

import paye.eoy.bizrules.helpers.BigDecimalUtil;
import paye.eoy.bizrules.helpers.WholePoundsHelper;
import paye.eoy.bizrules.p14.AbstractP14BusinessRule;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.Attribute;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;

/**
 * In the 2011/12 rules the Earnings Threshold (ET) became the Primary Threshold
 * (PT) so the names of the XML elements and rules have changed.
 * 
 * @author Doug
 * @since 13/7/2010
 */
public class EarningsLELtoPTRule extends AbstractP14BusinessRule {

    public static final String MUST_BE_ZERO_FOR_NI_CATEGORY = "6";

    private final String[] subcodes = { MUST_BE_ZERO_FOR_NI_CATEGORY, DocumentError.FORMAT, DocumentError.BLANK,
            DocumentError.MANDATORY, DocumentError.MINVALUE, DocumentError.MAXVALUE, DocumentError.UNKNOWN,
            DocumentError.ATTRIBUTE, DocumentError.VALUE, WholePoundsHelper.NOT_WHOLE_POUNDS };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    private static final BigDecimal ZERO = new BigDecimal(0.0);

    private BigDecimal lelToPT = BigDecimalUtil.ZERO;

    private String niCategory = "";

    public void setLELtoPT(final String value) {
        lelToPT = BigDecimalUtil.toBigDecimal(value);
    }

    public void setNICs(final Attribute[] attribute) {
        for (final Attribute element : attribute) {
            if ("Tab".equals(element.getLocalName())) {
                final String value = element.getValue();

                niCategory = value;
            }
        }
    }

    @Override
    protected final void initializeValueBindings() {

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/NICs", "NICs");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/NICs", "NICs");

        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/NICs/Earn/LELtoPT", "LELtoPT");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/NICs/Earn/LELtoPT", "LELtoPT");

    }

    public void processRule() {
        if (niCategory.length() > 0 && getNICategoryForWhichValueMustBeZero().indexOf(niCategory) != -1) {
            if (lelToPT.compareTo(ZERO) != 0) {
                raiseError(MUST_BE_ZERO_FOR_NI_CATEGORY, lelToPT);
            }
        }
    }

    protected String getNICategoryForWhichValueMustBeZero() {
        return "X";
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected void reset() {
        lelToPT = BigDecimalUtil.ZERO;
        niCategory = "";
    }

    @Override
    protected final String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P14/NICs", "/IRenvelope/EndOfYearReturn/P14Amended/NICs" };
    }

    @Override
    protected final void initializeEventBindings() {
        // do nothing
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        final String[] errorHandlers = new String[] { "distinguishWholePoundsError", "handleError" };
        binder.add("/IRenvelope/EndOfYearReturn/P14/NICs/Earn/LELtoPT", errorHandlers);
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/NICs/Earn/LELtoPT", errorHandlers);
    }
}
