/*
 * Created on May 25, 2005
 * 
 * $Id: AbstractNICategoryRule.java,v 1.5 2006/08/10 15:19:16 dec Exp $
 */
package paye.eoy.bizrules.p14.nic;

import java.util.HashMap;

import paye.channel.SubmissionChannel;
import paye.eoy.bizrules.p14.AbstractP14BusinessRule;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.Attribute;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;
import bizrules.binding.ValueBinder;

/**
 * @author Rob
 */
public class NICategoryRule extends AbstractP14BusinessRule {

    public static final String TOO_LONG = "toolong";

    public static final String BAD_TABLE_CODE = "badtablecode";

    public static final String TOO_SHORT = "tooshort";

    private String niCategory = "";

    private final String[] subcodes = { TOO_LONG, BAD_TABLE_CODE, TOO_SHORT };

    protected final static HashMap<ActiveTaxYear, String> validNICategoriesByYearNonPaper = new HashMap<ActiveTaxYear, String>();

    protected final static HashMap<ActiveTaxYear, String> validNICategoriesByYearPaper = new HashMap<ActiveTaxYear, String>();

    static {

        for (ActiveTaxYear year : TaxYear.allActive()) {
            validNICategoriesByYearNonPaper.put(year, "ABCDEJLNOQRTWX");
        }

        // Valid categories for Paper submissions
        for (ActiveTaxYear year : TaxYear.allActive()) {
            validNICategoriesByYearPaper.put(year, "ABCDEFGHIJKLMNOPQRSTUVWXYZ ");
        }
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public NICategoryRule() {
        setRuleName("NICategoryRule");
    }

    @Override
    protected void reset() {
        niCategory = "";
    }

    public void setNiCategory(final Attribute[] attribute) {
        for (final Attribute element : attribute) {
            if ("Tab".equals(element.getLocalName())) {
                final String data = element.getValue();
                niCategory = data;
            }
        }
    }

    @Override
    protected void initializeValueBindings() {
        final ValueBinder binder = getValueBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P14/NICs", "NiCategory");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/NICs", "NiCategory");
    }

    private boolean hasNICCategory() {
        return niCategory != null && niCategory.length() > 0;
    }

    public void processRule() {
        if (hasNICCategory()) {
            if (niCategory.length() > 1) {
                raiseError(TOO_LONG, niCategory);
            } else if (getValidNICategories().indexOf(niCategory) == -1) {
                raiseError(BAD_TABLE_CODE, niCategory);
            }
        } else {
            raiseError(TOO_SHORT);
        }
        niCategory = "";
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P14/NICs/@Tab", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/NICs/@Tab", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14/NICs", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/NICs", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended", "handleError");
    }

    @Override
    public boolean handleError(final DocumentError documentError) {
        boolean result = false;

        if (documentError.getSubCode().equals(DocumentError.ATTRIBUTE)) {
            // Suppress attribute errors and deal with them in the rule.
            result = true;
        }

        return result;
    }

    @Override
    protected final String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P14Amended/NICs", "/IRenvelope/EndOfYearReturn/P14/NICs" };
    }

    @Override
    protected final void initializeEventBindings() {
        // do nothing
    }

    protected final String getValidNICategories() {
        TaxYear year = getTaxYear();

        if (getSubmissionChannel() == SubmissionChannel.Paper) {
            // Some tax years distinguish between paper and non-paper channels
            String validNICategoriesForPaper = validNICategoriesByYearPaper.get(year);
            if (validNICategoriesForPaper != null) {
                return validNICategoriesForPaper;
            }
        }

        return validNICategoriesByYearNonPaper.get(year);
    }
}
