/*
 * Created on May 1, 2005
 *
 * $Id: AbstractNINORule.java,v 1.5 2006/06/15 10:29:25 dec Exp $
 */
package paye.eoy.bizrules.p14.nino;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import paye.channel.SubmissionChannel;
import paye.eoy.bizrules.p14.AbstractP14BusinessRule;
import paye.eoy.types.ReturnType;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;

/**
 * @author Rob
 */
public abstract class AbstractNINORule extends AbstractP14BusinessRule {

    public static final String NINO_RULE = "NINORule";

    public static final String SUFFIX = "suffix";

    public static final String PREFIX = "prefix";

    public static final String LENGTH = "length";

    public static final String PAPER_FORMAT = "paper format";

    private final String[] subcodes = { SUFFIX, PREFIX, DocumentError.FORMAT, DocumentError.BLANK, LENGTH };

    private static final Pattern PAPER_PATTERN = Pattern.compile("[A-Za-z]{2}[0-9]{6}[A-Z ]");

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public AbstractNINORule() {
        setRuleName(NINO_RULE);
    }

    public final void processRule() {
        final String nino = getNino();

        // NINO is optional (with conditions checked elsewhere) so no error if
        // it is missing
        if (isNinoPresent()) {
            if (getSubmissionChannel() == SubmissionChannel.Paper) {
                // Format is relaxed for Paper for all years - simply char 1&2
                // are alpha, 3-8 numeric and 9 A-Z
                final Matcher ninoMatcher = PAPER_PATTERN.matcher(nino);
                if (!ninoMatcher.matches()) {
                    raiseError(PAPER_FORMAT, nino);
                }
            } else {
                // NINO should be 9 characters exactly, even if the last
                // character
                // is a space
                if (nino.length() != 9) {
                    if (!nino.equals("")) {
                        raiseError(LENGTH, nino);
                    } else {
                        raiseError(DocumentError.BLANK);
                    }
                } else {
                    final NINOChecker checker = ninoChecker(nino);

                    // Any of these three checks will produce a 5012 error
                    if (!alphaPrefix(checker)) {
                        raiseError(DocumentError.FORMAT, nino);
                    } else if (!validSuffix(checker)) {
                        raiseError(SUFFIX, nino);
                    } else if (!validMiddle(checker)) {
                        raiseError(DocumentError.FORMAT, nino);
                    } else if (!validPrefix(checker)) {
                        // The prefix check is a business rule so only do it if
                        // the format is correct.
                        raiseError(PREFIX, nino);
                    }
                }
            }
        }
    }

    /**
     * Provide a NINOChecker instance suitable for the tax year that this rule is being
     * run for
     */
    protected abstract NINOChecker ninoChecker(String nino);

    private boolean alphaPrefix(final NINOChecker checker) {
        return checker.alphaPrefix();
    }

    private boolean validSuffix(final NINOChecker checker) {
        return checker.validSuffix();
    }

    private boolean validPrefix(final NINOChecker checker) {
        return checker.validPrefix();
    }

    private boolean validMiddle(final NINOChecker checker) {
        return checker.validMiddlePart();
    }

    @Override
    protected final ReturnType[] getActiveReturnTypes() {
        return new ReturnType[] { ReturnType.ORIGINAL, ReturnType.AMENDED };
    }

    @Override
    protected final String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P14Amended", "/IRenvelope/EndOfYearReturn/P14" };
    }

    @Override
    protected final void reset() {
        // do nothing
    }

    @Override
    protected final void initializeEventBindings() {
        // do nothing
    }

    @Override
    protected final void initializeValueBindings() {
        // do nothing
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/NINO", "ignoreError");
        binder.add("/IRenvelope/EndOfYearReturn/P14/NINO", "ignoreError");
    }

    @Override
    public boolean ignoreError(final DocumentError documentError) {
        // Suppress the DocumentErrors. We do all the checks in the rule itself.
        return true;
    }

}
