/*
 * Created on May 27, 2005
 *
 * $Id: FinalTaxCodeRule.java,v 1.3 2006/06/09 11:15:06 dec Exp $
 *
 */
package paye.eoy.bizrules.p14.payandtax;

import java.math.BigDecimal;

import paye.channel.SubmissionChannel;
import paye.eoy.bizrules.helpers.BigDecimalUtil;
import paye.eoy.bizrules.p14.AbstractP14BusinessRule;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;

/**
 * @author Rob
 */
public final class FinalTaxCodeRule extends AbstractP14BusinessRule {

    public static final String SVR_PRESENT = "svr present";

    public static final String NO_CODE = "no code";

    public static final String BAD_W1M1IND = "bad W1M1Ind";

    public static final String BAD_TAX_REGIME = "bad tax regime";

    private String taxCode;

    private BigDecimal tax;

    private final String[] subcodes = {NO_CODE, DocumentError.FORMAT, DocumentError.ATTRIBUTE, DocumentError.VALUE,
            DocumentError.MANDATORY, DocumentError.BLANK, DocumentError.INVALID, SVR_PRESENT, BAD_W1M1IND, BAD_TAX_REGIME};

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public void setTaxCode(final String value) {
        taxCode = value;
    }

    public void setTax(final String value) {
        tax = BigDecimalUtil.toBigDecimal(value);
    }

    @Override
    protected void initializeValueBindings() {
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp/Code", "taxCode");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp/Code", "taxCode");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp/Tax", "tax");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp/Tax", "tax");
    }

    public void processRule() {
        if (getSubmissionChannel() == SubmissionChannel.Paper) {
            /*
             * For paper channel submissions, format rule is relaxed to just
             * being a Character Set A check.
             */
            checkCharacterSetA(taxCode);
        }

        if (tax.compareTo(BigDecimalUtil.ZERO) != 0 && "".equals(taxCode)) {
            // Code is optional for Paper
            if (getSubmissionChannel() != SubmissionChannel.Paper) {
                raiseError(NO_CODE, taxCode);
            }
        }
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[]{"/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp",
                "/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp"};
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp/Code", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp/Code", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp/Code/@W1M1Ind", "badW1M1Ind");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp/Code/@W1M1Ind", "badW1M1Ind");
        binder.add("/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp/Code/@SVR", "svrPresent");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp/Code/@SVR", "svrPresent");
        binder.add("/IRenvelope/EndOfYearReturn/P14/PayAndTax/ThisEmp/Code/@TaxRegime", "badTaxRegime");
        binder.add("/IRenvelope/EndOfYearReturn/P14Amended/PayAndTax/ThisEmp/Code/@TaxRegime", "badTaxRegime");
    }

    public boolean badW1M1Ind(final DocumentError documentError) {
        raiseError(BAD_W1M1IND);
        return true;
    }

    public boolean svrPresent(final DocumentError documentError) {
        raiseError(SVR_PRESENT);
        return true;
    }

    public boolean badTaxRegime(final DocumentError documentError) {
        if (getTaxYear().isSince(ActiveTaxYear.TAX_YEAR_1920)) {
            raiseError(BAD_TAX_REGIME);
            return true;
        } else {
            return false;
        }
    }

    @Override
    protected void reset() {
        taxCode = "";
        tax = BigDecimalUtil.ZERO;
    }

    @Override
    public boolean handleError(final DocumentError documentError) {
        /*
         * Paper channel has a more relaxed format requirement to the FBI/EDI
         * channel so suppress the error for paper and we'll check it in the
         * rule.
         */
        if (getSubmissionChannel() == SubmissionChannel.Paper
                && documentError.getSubCode().equals(DocumentError.FORMAT)) {
            return true;
        } else {
            return super.handleError(documentError);
        }
    }

}
