/*
 * Created on May 24, 2005
 *
 * $Id: ECONFormatRule.java,v 1.2 2006/04/11 17:19:13 dec Exp $
 */
package paye.eoy.bizrules.p35;

import java.math.BigInteger;

import paye.channel.SubmissionChannel;
import paye.eoy.bizrules.AbstractPAYEBusinessRule;
import paye.eoy.bizrules.helpers.Mod19;
import paye.eoy.types.ReturnType;
import paye.eoy.types.SubmissionType;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;
import bizrules.binding.ValueBinder;

/**
 * @author doug
 * @since May 24, 2005
 */
public class ECONFormatRule extends AbstractPAYEBusinessRule {
    private final String[] subcodes = { LENGTH, MOD19, PREFIX, CHAR_9_MUST_BE_LETTER, CHAR_2_TO_8_FORMAT };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public static final String LENGTH = "length";

    public static final String MOD19 = "mod19";

    public static final String PREFIX = "prefix";

    public static final String CHAR_9_MUST_BE_LETTER = "char9";

    public static final String CHAR_2_TO_8_FORMAT = "char 2 to 8 format";

    private static BigInteger minNonPaper = new BigInteger("3000000");

    private static BigInteger maxNonPaper = new BigInteger("3999999");

    private static BigInteger minPaper = new BigInteger("0000000");

    private static BigInteger maxPaper = new BigInteger("9999999");

    private String econ;

    public ECONFormatRule() {
        setRuleName("ECONRule");
    }

    public void setEcon(final String value) {
        econ = value;
    }

    public void processRule() {
        // According to Case 1056, paper submissions should be less strictly
        // checked.
        SubmissionChannel channel = getSubmissionChannel();
        BigInteger min = (channel == SubmissionChannel.Paper ? minPaper : minNonPaper);
        BigInteger max = (channel == SubmissionChannel.Paper ? maxPaper : maxNonPaper);

        if (econ != null) {
            if (econ.length() != 9) {
                raiseError(LENGTH, econ);
            } else {
                final char firstChar = econ.charAt(0);
                final String numericPartString = econ.substring(1, 8);
                final char finalChar = econ.charAt(8);

                try {
                    final BigInteger numericPart = new BigInteger(numericPartString);
                    if (numericPart.compareTo(min) < 0 || numericPart.compareTo(max) > 0) {
                        raiseError(CHAR_2_TO_8_FORMAT, econ);
                    } else if (firstChar != 'E') {
                        raiseError(PREFIX, econ);
                    } else if (!Character.isLetter(finalChar)) {
                        raiseError(CHAR_9_MUST_BE_LETTER, econ);
                    } else if (channel != SubmissionChannel.Paper && Mod19.table.indexOf(finalChar) == -1) {
                        raiseError(MOD19, econ);
                    } else if (channel != SubmissionChannel.Paper && Mod19.calcECON(numericPartString) != finalChar) {
                        raiseError(MOD19, econ);
                    }
                } catch (final NumberFormatException nfe) {
                    raiseError(CHAR_2_TO_8_FORMAT, econ);
                }
            }
        }
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn/P35Amended/EmployerDetails",
                "/IRenvelope/EndOfYearReturn/P35/EmployerDetails" };
    }

    @Override
    protected SubmissionType[] getActiveSubmissionTypes() {
        return new SubmissionType[] { SubmissionType.P35PART, SubmissionType.COMPLETE };
    }

    @Override
    protected ReturnType[] getActiveReturnTypes() {
        return new ReturnType[] { ReturnType.ORIGINAL, ReturnType.AMENDED };
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected void initializeValueBindings() {
        final ValueBinder binder = getValueBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P35/EmployerDetails/ECON", "econ");
        binder.add("/IRenvelope/EndOfYearReturn/P35Amended/EmployerDetails/ECON", "econ");
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();

        binder.add("/IRenvelope/EndOfYearReturn/P35/EmployerDetails/ECON", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P35Amended/EmployerDetails/ECON", "handleError");
    }

    @Override
    public boolean handleError(final DocumentError error) {
        if (error.getSubCode().equals(DocumentError.FORMAT)) {
            /*
             * We're actually duplicating the schema format checks in
             * processRule so if we see a schema format error for the ECON we
             * can simply suppress it.
             */
            return true;
        } else {
            return false;
        }
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

}
