/*
 * Created on May 31, 2005
 *
 * $Id: TotalNICRuleComplete.java,v 1.2 2006/08/10 15:19:16 dec Exp $
 * 
 */
package paye.eoy.bizrules.p35;

import java.math.BigDecimal;

import paye.eoy.bizrules.AbstractPAYEBusinessRule;
import paye.eoy.bizrules.helpers.BigDecimalTotalizer;
import paye.eoy.bizrules.helpers.BigDecimalUtil;
import paye.eoy.types.SubmissionType;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;
import xmldoc.DocumentError;
import bizrules.binding.ErrorBinder;
import bizrules.errors.MonetaryDiscrepancy;

/**
 * Cross-check the total NIC in the P35 against the sum of the NIC values in the
 * P14s.
 */
public class TotalNICRuleComplete extends AbstractPAYEBusinessRule {

    public static final String WRONG_TOTAL = "bad total";

    private BigDecimal totalNIC = BigDecimalUtil.ZERO;

    private boolean totalNICSet = false;

    private final BigDecimalTotalizer totalizer = new BigDecimalTotalizer();

    private final String[] subcodes = { WRONG_TOTAL, DocumentError.FORMAT, DocumentError.MANDATORY, DocumentError.BLANK };

    @Override
    public String[] getSubcodes() {
        return subcodes;
    }

    public TotalNICRuleComplete() {
        setRuleName("TotalNICRule");
    }

    @Override
    protected void initializeValueBindings() {
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14/NICs/Both", "P14Nic");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P14Amended/NICs/Both", "P14Nic");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/NIC/TotalNIC", "TotalNic");
        getValueBinder().add("/IRenvelope/EndOfYearReturn/P35/Summary/NIC/TotalNIC", "TotalNic");
    }

    public void setP14Nic(final String value) {
        totalizer.add(value);
    }

    public void setTotalNic(final String value) {
        if (!value.equals("")) {
            totalNIC = BigDecimalUtil.toBigDecimal(value);
            totalNICSet = true;
        }
    }

    @Override
    protected ActiveTaxYear[] getApplicableTaxYears() {
        return TaxYear.allActive();
    }

    @Override
    protected SubmissionType[] getActiveSubmissionTypes() {
        final SubmissionType[] activeTypes = new SubmissionType[] { SubmissionType.COMPLETE };
        return activeTypes;
    }

    @Override
    protected String[] getInvocationPoints() {
        return new String[] { "/IRenvelope/EndOfYearReturn" };

    }

    public void processRule() {
        final BigDecimal calculated = totalizer.getTotal();

        /**
         * If the NIC hasn't been flagged as being set then the field must have
         * been blank or empty so we will get a schema error. No need to also
         * complain that the total isn't right.
         */
        if (totalNICSet && calculated.compareTo(totalNIC) != 0) {
            raiseQS3Error(WRONG_TOTAL, new MonetaryDiscrepancy(totalizer.getTotal(), totalNIC));
        }
    }

    @Override
    protected void initializeEventBindings() {
        // do nothing
    }

    @Override
    protected void initializeErrorBindings() {
        final ErrorBinder binder = getErrorBinder();
        binder.add("/IRenvelope/EndOfYearReturn/P35/Summary/NIC/TotalNIC", "handleError");
        binder.add("/IRenvelope/EndOfYearReturn/P35Amended/Summary/NIC/TotalNIC", "handleError");
    }
}
