package paye.eoy.output.mapper;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import paye.channel.SubmissionChannel;
import paye.multiyear.ActiveTaxYear;
import paye.multiyear.TaxYear;

import com.gsl.logging.LoggerFactory;

import ericResponse.output.error.ERErrorDetail;
import ericResponse.output.error.EricErrorType;
import ericResponse.output.error.EricResponseErrors;
import ericResponse.output.translator.TranslationContext;
import ericResponse.output.translator.mapper.ErrorMapper;

public abstract class AbstractEOYErrorMapper implements ErrorMapper {

    protected static final Logger LOGGER = LoggerFactory.getLogger("paye.eoy.output.mapper");

    //protected static final Logger LOGGER = LoggerFactory.getLogger("ericresponse.mapping");

    private final Map<TaxYear, Map<String, ERErrorDetail>> errorDetailMaps = new HashMap<TaxYear, Map<String, ERErrorDetail>>();

	private String name;

    public AbstractEOYErrorMapper() {
        setupName();
    }

    protected Map<String, ERErrorDetail> getErrorDetailMap(final TranslationContext context) {
        final TaxYear year = getTaxYearFromTranslationContext(context);
        final SubmissionChannel channel = getSubmissionChannelFromTranslationContext(context);

        Map<String, ERErrorDetail> errorDetailMap = errorDetailMaps.get(year);
        if (errorDetailMap == null) {
            errorDetailMap = new HashMap<String, ERErrorDetail>();

            final String[][] subCodeData = getErrorMapData(channel, year);

            final Properties messageProperties = getMessageProperties(year);

            for (final String[] subCodeDataRow : subCodeData) {
                final String subCode = subCodeDataRow[0];
                final ERErrorDetail detail = buildDetail(year, subCodeDataRow, messageProperties);

                if (errorDetailMap.containsKey(subCode)) {
                    LOGGER.log(Level.SEVERE, "Duplicate sub-code in eric error mapper: " + getClass().getName());
                }

                errorDetailMap.put(subCode, detail);
            }

            mergeInGenericMappings(year, errorDetailMap, messageProperties);

            errorDetailMaps.put(year, errorDetailMap);
        }
        return errorDetailMap;
    }

    /**
     * Default is just to return the common set of map data. Subclasses can
     * override if need be
     */
    protected String[][] getErrorMapData(final SubmissionChannel channel, final TaxYear taxYear) {
        return getErrorMapData();
    }

    protected String[][] getErrorMapData() {
        throw new UnsupportedOperationException(
                "Subclass must override getErrorMapData() or getErrorMapData(final SubmissionChannel channel, final TaxYear taxYear)");
    }

    protected abstract Properties getMessageProperties(TaxYear taxYear);

    protected void mergeInGenericMappings(TaxYear taxYear, final Map<String, ERErrorDetail> errorDetailMap, final Properties messageProperties) {
        // do nothing here by default
    }

    protected ERErrorDetail buildDetail(TaxYear taxYear, final String[] row, final Properties messageProperties) {
        ERErrorDetail detail = null;
        try {
            if (row[1].equals(EricResponseErrors.DO_NOT_MAP)) {
                detail = EricResponseErrors.DO_NOT_MAP_DETAIL;
            } else {
                detail = new ERErrorDetail();
                detail.setLocator(getLocator(taxYear, row[0]));

                int i = 1;
                detail.setCode(row[i++]);

                final String brCode = row[i++];
                detail.setBRCode(brCode);

                if (i < row.length) {
                    detail.setErrorType(EricErrorType.fromString(row[i++]));
                }

                if (messageProperties != null) {
                    final String text = messageProperties.getProperty(brCode);
                    if (text != null && text.trim().length() > 0) {
                        detail.setMessageText(text);
                    }
                }
            }
        } catch (final IndexOutOfBoundsException e) {
            LOGGER.log(Level.SEVERE, "Malformed data in eric error mapper: " + getClass().getName());
        }
        return detail;
    }

    public boolean isActive(final TranslationContext context) {
        final TaxYear year = getTaxYearFromTranslationContext(context);
        final ActiveTaxYear[] taxYears = getActiveTaxYears();
        for (final TaxYear element : taxYears) {
            if (year == element) {
                return true;
            }
        }
        return false;
    }

    protected abstract ActiveTaxYear[] getActiveTaxYears();

    protected TaxYear getTaxYearFromTranslationContext(final TranslationContext context) {
        return ((PAYETranslationContext) context).getTaxYear();
    }

    protected SubmissionChannel getSubmissionChannelFromTranslationContext(final TranslationContext context) {
        return ((PAYETranslationContext) context).getChannel();
    }

    protected String getLocator(String subCode) {
    	throw new UnsupportedOperationException(
				"Subclass must override getLocator(String subCode) or getLocator(final TaxYear taxYear, String subCode)");
    }
	
	protected String getLocator(final TaxYear taxYear, String subCode) {
		return getLocator(subCode);
	}

	public ERErrorDetail mapError(final TranslationContext context, final String ruleName, final String subCode) {
	    if (isActive(context)) {
	        final Map<String, ERErrorDetail> errorDetailMap = getErrorDetailMap(context);
	        if (errorDetailMap != null) {
	            final ERErrorDetail detail = (ERErrorDetail) errorDetailMap.get(subCode);
	
	            if (detail != null && detail != EricResponseErrors.DO_NOT_MAP_DETAIL && detail.getMessageText().length() == 0) {
	                LOGGER.log(Level.SEVERE, "Eric response message not found: " + detail);
	            }
	
	            return detail;
	        }
	    }
	    return null;
	}

	public Set<String> getDeclaredSubcodes() {
	    final Set<String> mappedCodesSet = new HashSet<String>();
	    final String[][] errorMapData = getErrorMapData();
	    for (final String[] dataRow : errorMapData) {
	        mappedCodesSet.add(dataRow[0]);
	    }
	    return mappedCodesSet;
	}

	public String getName() {
	    return name;
	}

	/**
	 * By default we use the class name of the concrete class that extends us as
	 * the rule name. This can be overridden by the subclass if required.
	 */
	protected void setupName() {
	    this.name = this.getClass().getSimpleName();
	}
}
