/**
 * 
 */
package paye.eoy.validator;

import java.util.List;

import paye.tolerances.Tolerable;
import paye.tolerances.Tolerance;
import processingError.ProcessingError;
import processingError.ProcessingErrorCollector;
import ericResponse.output.EricResponsePostValidationDelegate;
import ericResponse.output.error.EricResponseError;

/**
 * This error collector will check to see if the ProcessingError being added
 * maps to an EricResponseError with a tolerance of DNA. If so, it drops the
 * error rather than adding it to the list.
 * 
 * This collector wraps another ProcessingErrorCollector and delegates all other
 * operations to that collector.
 * 
 * I wrote this as part of the solution to <a
 * href='https://gsl.fogbugz.com/default.asp?100'>bug 100</a>. This lets us
 * filter DNA's at source and so avoid adding a lot of errors to the list which
 * might be filtered out later, potentially exceeding the maximum collection
 * count with DNAs prior to encountering a non-DNA error. In that case the
 * submission would be accepted on tolerance even though it should fail.
 * 
 * @author douglas
 * 
 */
public class DNAFilteringErrorCollector implements ProcessingErrorCollector {
    private final EricResponsePostValidationDelegate delegate;

    private long dnaCount = 0;

    private final ProcessingErrorCollector wrappedCollector;

    public DNAFilteringErrorCollector(final ProcessingErrorCollector wrappedCollector,
            final EricResponsePostValidationDelegate delegate) {
        this.wrappedCollector = wrappedCollector;
        this.delegate = delegate;
    }

    public void addError(final ProcessingError pe) {
        final EricResponseError ere = delegate.mapError(pe);

        if (ere instanceof Tolerable) {
            final Tolerable t = (Tolerable) ere;
            if (t.getTolerance() != Tolerance.DNA) {
                wrappedCollector.addError(pe);
            } else {
                /*
                 * Just used for internal accounting and as a convenient
                 * breakpoint
                 */
                dnaCount++;
            }
        } else {
            wrappedCollector.addError(pe);
        }
    }

    protected long getDNACount() {
        return dnaCount;
    }

    public void clearErrors() {
        wrappedCollector.clearErrors();
    }

    public void dumpErrorsNotOfType(final Class<? extends ProcessingError> clazz) {
        wrappedCollector.dumpErrorsNotOfType(clazz);
    }

    public Object getErrorExtraInfo() {
        return wrappedCollector.getErrorExtraInfo();
    }

    public List<ProcessingError> getErrors() {
        return wrappedCollector.getErrors();
    }

    public void setErrorExtraInfo(final Object extraInfo) {
        wrappedCollector.setErrorExtraInfo(extraInfo);
    }

    public void setMaxErrors(final int maxErrors) {
        wrappedCollector.setMaxErrors(maxErrors);
    }
}
