package paye.eoy.validator;

import hmrc.validator.HMRCValidationParams;
import paye.channel.SubmissionChannel;
import paye.multiyear.TaxYear;

/**
 * Extends the HMRC parameters base class and implements additional PAYE
 * specific validation parameters.
 * <p>
 * Create an instance of this class and set its parameters as required to
 * control various aspects of the PAYE document validation process. An instance
 * of this class should be passed to the PAYE validator when validating a PAYE
 * submission.
 * 
 * @see PAYEValidator
 */
public class PAYEValidationParams extends HMRCValidationParams {

    private TaxYear taxYear = TaxYear.TAX_YEAR_0405;

    private SubmissionChannel channel = SubmissionChannel.FBI;

    private boolean isToleranceCheckRequired = true;

    private boolean useZeroToleranceThresholds = false;

    private boolean injectMISResultIntoResponse = false;

    @Override
    public String toString() {
        String s = "PAYEValidationParams [";

        s += super.toString();
        s += ",\n";
        s += "taxYear=" + taxYear;
        s += ", ";
        s += "isToleranceCheckRequired=" + isToleranceCheckRequired;
        s += ", ";

        s += "]";
        return s;
    }

    /**
     * Overrides the default implementation to return a document variant object
     * based on the tax year that was set.
     * 
     * @see #setTaxYear(TaxYear)
     * 
     * @return An variant object based on the tax year that was set.
     */
    @Override
    protected Object getVariant() {
        return getTaxYear();
    }

    /**
     * Indicates whether or not to apply tolerance checks to any errors found in
     * the document being validated.
     * <p>
     * Defaults to true.
     * 
     * @return True if tolerance checks should be applied, false if not.
     */
    public boolean isToleranceCheckRequired() {
        return isToleranceCheckRequired;
    }

    /**
     * Sets whether or not to apply tolerance checks to any errors found in the
     * document being validated.
     * 
     * @param isToleranceCheckRequired
     *            true if tolerance checks should be applied, false if not.
     */
    public void setToleranceCheckRequired(final boolean isToleranceCheckRequired) {
        this.isToleranceCheckRequired = isToleranceCheckRequired;
    }

    /**
     * Set this to true to override the tolerance thresholds for the tax year
     * and use zero thresholds instead. This has the effect of disabling Type 2
     * and Cash tolerances but leaving DNA tolerances active (as long as the
     * toleranceCheckRequired is also set to true).
     * 
     * If set to false, and toleranceCheckRequired is true, then the thresholds
     * will be determined according to the tax year of the submission.
     */
    public void setUseZeroToleranceThresholds(final boolean useZeroToleranceThresholds) {
        this.useZeroToleranceThresholds = useZeroToleranceThresholds;
    }

    public boolean getUseZeroToleranceThresholds() {
        return useZeroToleranceThresholds;
    }

    /**
     * If this is set to true then the PAYEValidationResult data will be
     * injected into the EricErrorResponse XML as an Application element.
     */
    public void setInjectMISResultIntoResponse(final boolean injectResultIntoResponse) {
        this.injectMISResultIntoResponse = injectResultIntoResponse;
    }

    public boolean getInjectMISResultIntoResponse() {
        return injectMISResultIntoResponse;
    }

    /**
     * Gets the tax year that the input submission will be validator with
     * respect to.
     * 
     * @return The tax year that the input submission will be validator with
     *         respect to
     */
    public TaxYear getTaxYear() {
        return taxYear;
    }

    /**
     * Sets the tax year that the input submission will be validated with
     * respect to.
     * 
     * @see #setTaxYear(String)
     * 
     * @param taxYear
     *            The tax year to set.
     */
    public void setTaxYear(final TaxYear taxYear) {
        this.taxYear = taxYear;
    }

    /**
     * Sets the tax year that the input submission will be validated with
     * respect to, based on a string value of the form "0405", "0506", etc.
     * <p>
     * This is a short cut for:
     * <code>setTaxYear(TaxYear.fromString(taxYearString))</code>
     * 
     * @see #setTaxYear(TaxYear)
     * 
     * @param taxYearString
     *            The string to decode into the tax year to set.
     * 
     * @throws IllegalArgumentException
     *             if the taxYearString parameter cannot be decoded to a
     *             supported tax year.
     */
    public void setTaxYear(final String taxYearString) {
        setTaxYear(TaxYear.fromString(taxYearString));
    }

    public SubmissionChannel getChannel() {
        return channel;
    }

    public void setChannel(final SubmissionChannel channel) {
        this.channel = channel;
    }
}
