package paye.eoy.validator;

import hmrc.validator.ChRISEnvelopeValidatorProvider;
import hmrc.validator.ERICEnvelopeValidatorProvider;
import hmrc.validator.GatewayEnvelopeValidatorProvider;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.Properties;

import paye.channel.SubmissionChannel;

import com.gsl.docValidator.DocumentValidator;
import com.gsl.docValidator.DocumentValidatorServices;
import com.gsl.docValidator.ValidationException;
import com.gsl.docValidator.ValidationParams;
import com.gsl.hmrc.paye.paper.FixupReader;
import com.gsl.logging.LoggerFactory;
import com.gsl.reader.nscorrect.NamespaceCorrectingReader;

import ericResponse.output.EricResponsePostValidationDelegate;

/**
 * To validate a document, instantiate this class and call
 * validateSubmission(...) passing the required parameters.
 * 
 * @see PAYEValidationParams
 */
public class PAYEValidator extends DocumentValidator {

    // this is the message text to use for failed submissions
    private static final String SUBMISSION_FAILED_MSG_TEXT = "The submission of this document has failed due to departmental specific business logic in the Body tag.";

    private PAYEValidationParams payeValidationParams;

    private EricResponsePostValidationDelegate delegate;

    /**
     * This is the entry point into this validator implementation. Clients may
     * instantiate this class and call this method to validate a submission.
     * 
     * @see PAYEValidationParams
     * 
     * @param input
     *            The input XML submission will be read from this Reader
     *            instance.
     * @param output
     *            The ERIC response XML will be written to this Writer instance,
     *            if it is non-null.
     * @param params
     *            The parameters that control various aspects of the validation
     *            process.
     * 
     * @return The validation result object containing information about the
     *         result of the validating the input document.
     * 
     * @throws ValidationException
     *             Thrown if there was an error encountered at any stage during
     *             validation.
     */
    public synchronized PAYEValidationResult validateSubmission(Reader input, final Writer output, final PAYEValidationParams params)
            throws ValidationException {

        this.payeValidationParams = params;
        this.delegate = new PAYEValidationDelegate(params, output);

        params.getResponseParams().setFailureMessage(SUBMISSION_FAILED_MSG_TEXT);

        // set logging levels
        LoggerFactory.setLevels(params.getLoggingParams().getLogLevels());

        // add validator providers
        getController().addProvider(new GatewayEnvelopeValidatorProvider());
        getController().addProvider(new ChRISEnvelopeValidatorProvider());
        getController().addProvider(new ERICEnvelopeValidatorProvider());
        getController().addProvider(new PAYEValidatorProvider());

        /*
         * A bit of a hack but there doesn't seem to be an obviously better
         * place to get these two pieces of information together at the same
         * time.
         * 
         * This relates to Case 28 - if tolerances are switched on the it is
         * possible for a schema error to cause business rule validation to be
         * turned off, but then be filtered out at the tolerance check because
         * it was a DNA. By turning off the "ignore business rules following
         * document error" flag when tolerances are in forces we can avoid this,
         * but still support that flag in non-tolerance situations (such as the
         * Schematron validator)
         */
        if (params.isToleranceCheckRequired()) {
            params.getBizruleParams().setIgnoreBusinessRulesFollowingDocumentError(false);
        }

        if (params.getChannel() == SubmissionChannel.EDI) {
            input = new NamespaceCorrectingReader(input, params.getTaxYear());
        } else if (params.getChannel() == SubmissionChannel.Paper) {
            input = new FixupReader(input, params.getTaxYear());
        }

        // invoke the validator, casting and returning the result to the client
        return (PAYEValidationResult) getController().validate(this, input, params, delegate);
    }

    @Override
    protected DocumentValidatorServices createServices(final ValidationParams params) {
        return new PAYEValidatorServices(payeValidationParams, delegate);
    }

    @Override
    protected void validationComplete() {
        this.payeValidationParams = null;
        this.delegate = null;
    }

    public static String getBusinessRulesVersion() {
        String version = "";
        String date = "";
        String rev = "";
        final Properties properties = new Properties();

        try {
            properties.load(PAYEValidationResult.class.getResourceAsStream("version.properties"));
            version = properties.getProperty("version");
            date = properties.getProperty("date");
            rev = properties.getProperty("rev");
        } catch (final IOException e) {
            version = "unknown";
            date = "unknown";
        }

        date = date.replaceFirst("\\$LastChangedDate: ", "");
        date = date.replaceFirst(" \\$", "");
        date = date.replaceAll(".*\\((.*)\\)", "$1");

        rev = rev.replaceFirst("\\$LastChangedRevision: ", "Rev: ");
        rev = rev.replaceFirst(" \\$", "");

        return "Version " + version + ", " + rev + ", " + date;
    }
}
