package paye.eoy.validator;

import govtalk.header.GovTalkHeaderDetails;
import hmrc.validator.ChRISEnvelopeValidatorProvider;
import hmrc.validator.DelegateHandler;
import hmrc.validator.ERICEnvelopeValidatorProvider;
import hmrc.validator.GatewayEnvelopeValidatorProvider;
import hmrc.validator.HMRCValidationDelegate;
import hmrc.validator.HMRCValidationResultInterface;
import hmrc.validator.PAYEDetailCollector;
import hmrc.validator.StandardIREnvelopeCollector;
import hmrc.validator.result.ValidationStatus;
import paye.channel.SubmissionChannel;
import paye.eoy.bizrules.PAYEIRNamespaceMapping;
import paye.eoy.bizrules.PAYESubmissionPropertyCollector;
import paye.eoy.bizrules.header.PAYEPeriodEndRule;
import paye.eoy.bizrules.p14.P14IdentifierCollector;
import paye.eoy.output.mapper.PAYETranslationContext;
import paye.multiyear.TaxYear;
import xmldoc.util.ClasspathEntityResolver;
import xmldoc.util.EntityResolver;

import com.gsl.docValidator.DocumentValidatorImplementor;
import com.gsl.docValidator.ValidationParams;
import com.gsl.docValidator.ValidatorProvider;

import ericResponse.output.PAYESubmissionDetails;
import ericResponse.output.PAYESuccessResponseGenerator;
import ericResponse.output.SuccessResponseGenerator;
import ericResponse.output.error.EricResponseErrorFactory;
import ericResponse.output.filters.FilterBizErrorsIfDocErrors;
import ericResponse.output.translator.TranslationContext;

public class PAYEValidatorProvider implements ValidatorProvider, DelegateHandler {

    // this is the message text to use for failed submissions
    private static final String SUBMISSION_FAILED_MSG_TEXT = "The submission of this document has failed due to departmental specific business logic in the Body tag.";

    private TaxYear taxYear;

    private SubmissionChannel channel;

    private SubmissionInfo submissionInfo;

    private PAYESubmissionDetails payeSubmissionDetails;

    public EntityResolver getEntityResolver(final ValidationParams params) {
        final ClasspathEntityResolver entityResolver = new ClasspathEntityResolver("PAYEValidator",
                PAYEValidatorConstants.EOY_ENTITY_RESOLUTION_TABLE);

        for (String[] consNSMapping : PAYEValidatorConstants.CONSOLIDATED_RESOLUTION_TABLE) {
            entityResolver.addTableEntry(consNSMapping[0], consNSMapping[1]);
        }
        return entityResolver;
    }

    public void init(final DocumentValidatorImplementor validator) {
        final TaxYear year = ((PAYEValidationParams) validator.getParams()).getTaxYear();

        GatewayEnvelopeValidatorProvider.addIRNamespace(PAYEIRNamespaceMapping.getInstance().getIRNamespaceForTaxYear(year));

        ChRISEnvelopeValidatorProvider.addIRNamespace(PAYEIRNamespaceMapping.getInstance().getIRNamespaceForTaxYear(year));
        // Consolidated submissions only ever come from ChRIS
        for (String[] consNSMapping : PAYEValidatorConstants.CONSOLIDATED_RESOLUTION_TABLE) {
            ChRISEnvelopeValidatorProvider.addIRNamespace(consNSMapping[0]);
        }

        ERICEnvelopeValidatorProvider.addIRNamespace(PAYEIRNamespaceMapping.getInstance().getIRNamespaceForTaxYear(year));
    }

    private boolean isConsolidatedNamespace(String ns) {
        for (String[] consNSMapping : PAYEValidatorConstants.CONSOLIDATED_RESOLUTION_TABLE) {
            if (consNSMapping[0].equals(ns))
                return true;
        }
        return false;
    }

    public static boolean isEOYNamespace(String namespace) {
        for (String[] eoyNSMapping : PAYEValidatorConstants.EOY_ENTITY_RESOLUTION_TABLE) {
            if (eoyNSMapping[0].equals(namespace))
                return true;
        }
        return false;
    }

    public void provisionNamespace(final String namespace, final DocumentValidatorImplementor validator) {
        if (isEOYNamespace(namespace) || isConsolidatedNamespace(namespace)) {
            final PAYEValidationParams params = (PAYEValidationParams) validator.getParams();
            params.getResponseParams().setFailureMessage(SUBMISSION_FAILED_MSG_TEXT);

            TaxYear.setIRNamespaceMapping(PAYEIRNamespaceMapping.getInstance());
            taxYear = params.getTaxYear();
            channel = params.getChannel();

            final PAYEValidationDelegate delegate = (PAYEValidationDelegate) validator.getDelegate();
            delegate.setDelegateHandler(this);

            final StandardIREnvelopeCollector irEnvelopeCollector = new StandardIREnvelopeCollector(namespace);
            validator.addElementProcessor(irEnvelopeCollector);

            final SubmissionInfoCollector submissionInfoCollector = new SubmissionInfoCollector(taxYear);
            validator.addElementProcessor(submissionInfoCollector);

            final PAYEDetailCollector payeDetailCollector = new PAYEDetailCollector(namespace);
            validator.addElementProcessor(payeDetailCollector);

            validator.addElementProcessor(new PAYESubmissionPropertyCollector(params.getChannel()));
            validator.addElementProcessor(new P14IdentifierCollector());

            validator.addBusinessRules("paye.eoy.bizrules");
            validator.addBusinessRule(new PAYEPeriodEndRule(taxYear));

            this.submissionInfo = submissionInfoCollector;

            delegate.addErrorMappers("paye.eoy.output.mapper");
            delegate.addGenericMappers("paye.eoy.output.mapper.generic");

            final GovTalkHeaderDetails govTalkHeaderDetails = delegate.getGovTalkHeaderDetails();

            delegate.addResponseGenerator(new SuccessResponseGenerator(govTalkHeaderDetails, irEnvelopeCollector, false, false,
                    false));
            delegate.addResponseGenerator(new PAYESuccessResponseGenerator(irEnvelopeCollector, payeDetailCollector));

            delegate.setIREnvelopeDetails(irEnvelopeCollector);
            this.payeSubmissionDetails = payeDetailCollector;
        }
    }

    public void destroy() {
    }

    public HMRCValidationResultInterface createResult(final HMRCValidationDelegate delegate, final ValidationParams pparams) {

        final PAYEValidationParams params = (PAYEValidationParams) pparams;
        if (params.isToleranceCheckRequired()) {
            delegate.addErrorFilter(new ToleranceDNAFilter());
        }
        /**
         * Must add the biz rule filter after the DNA filter for proper
         * behaviour
         */
        if (params.getBizruleParams().getIgnoreBusinessRulesFollowingDocumentError()) {
            delegate.addErrorFilter(new FilterBizErrorsIfDocErrors());
        }

        final PAYEValidationResultBean result = new PAYEValidationResultBean();
        result.setTaxYear(taxYear);
        result.setChannel(channel);

        return result;
    }

    public void processResult(final HMRCValidationResultInterface rresult, final ValidationParams pparams) {
        final PAYEValidationResultBean result = (PAYEValidationResultBean) rresult;

        if (submissionInfo != null) {
            if (submissionInfo.getSubmissionType().isPartSubmission()) {
                result.getQS3Result().setValidationStatus(ValidationStatus.NOT_VALIDATED);
            }

            result.setP14Count(submissionInfo.getP14Count());
            result.setSubmissionType(submissionInfo.getSubmissionType());

        }

        result.setPayeSubmissionDetails(payeSubmissionDetails);
    }

    public EricResponseErrorFactory createEricResponseErrorFactory() {
        return new PAYEEricResponseErrorFactory();
    }

    public TranslationContext createTranslationContext() {
        return new PAYETranslationContext(taxYear, channel);
    }
}
