package paye.eoy.validator;

import paye.eoy.types.SubmissionType;
import paye.multiyear.TaxYear;
import bizrules.AbstractElementProcessor;
import bizrules.registry.DocumentEventDetail;

/**
 * Collects information about the SubmissionType, count of P14s and total
 * payable. These are used to prepare the validation result (including tolerance
 * checking)
 */
public class SubmissionInfoCollector extends AbstractElementProcessor implements SubmissionInfo {

    private int p14Count;

    private final TaxYear taxYear;

    private SubmissionType submissionType = SubmissionType.UNKNOWN;

    private double totalPayable;

    public SubmissionInfoCollector(final TaxYear taxYear) {
        this.taxYear = taxYear;
    }

    @Override
    protected void initializeValueBindings() {
        addValueBinding("/IRenvelope/EndOfYearReturn/SubmissionType", "submissionType");
        addValueBinding("/IRenvelope/EndOfYearReturn/P35/Summary/CombinedAmounts/TotalPayable", "totalPayable");
        addValueBinding("/IRenvelope/EndOfYearReturn/P35Amended/Summary/CombinedAmounts/TotalPayable", "totalPayable");
    }

    @Override
    protected void initializeEventBindings() {
        addEventBinding("/IRenvelope/EndOfYearReturn/P14", "onP14");
        addEventBinding("/IRenvelope/EndOfYearReturn/P14Amended", "onP14");
    }

    public void onP14(final DocumentEventDetail detail) {
        p14Count++;
    }

    public void setSubmissionType(final String value) {
        try {
            submissionType = SubmissionType.fromString(value);
        } catch (final IllegalArgumentException e) {
            submissionType = SubmissionType.UNKNOWN;
        }
    }

    public void setTotalPayable(final String value) {
        try {
            totalPayable = Double.parseDouble(value);
        } catch (final NumberFormatException e) {
            totalPayable = 0;
        }
    }

    public SubmissionType getSubmissionType() {
        return this.submissionType;
    }

    public int getP14Count() {
        return p14Count;
    }

    public double getTotalPayable() {
        return totalPayable;
    }

    @Override
    protected String[] getApplicableNamespaces() {
        return new String[] { taxYear.getIREnvelopeNamespaceString() };
    }

    public boolean isP35Part() {
        return getSubmissionType() == SubmissionType.P35PART;
    }

    public boolean isP14Part() {
        return getSubmissionType() == SubmissionType.P14PART;
    }

    public boolean isComplete() {
        return getSubmissionType() == SubmissionType.COMPLETE;
    }

    public boolean isUnknownSubmissionType() {
        return getSubmissionType() == SubmissionType.UNKNOWN;
    }
}
