/*
 * Created on 27-Nov-2005
 *
 * $Id: TaxYear.java,v 1.10 2006/05/17 08:58:54 dec Exp $
 */
package paye.multiyear;

import java.util.*;

/**
 * @author Douglas Clinton
 * @since 27-Nov-2005
 */
public abstract class TaxYear {

    public static final String TAX_YEAR = "TAX_YEAR";

    private final String yearString;

    private static final Map<String, TaxYear> taxYearMap = new HashMap<String, TaxYear>();

    public static final InactiveTaxYear TAX_YEAR_0304 = new InactiveTaxYear("0304");
    public static final InactiveTaxYear TAX_YEAR_0405 = new InactiveTaxYear("0405");
    public static final InactiveTaxYear TAX_YEAR_0506 = new InactiveTaxYear("0506");
    public static final InactiveTaxYear TAX_YEAR_0607 = new InactiveTaxYear("0607");
    public static final InactiveTaxYear TAX_YEAR_0708 = new InactiveTaxYear("0708");
    public static final InactiveTaxYear TAX_YEAR_0809 = new InactiveTaxYear("0809");
    public static final InactiveTaxYear TAX_YEAR_0910 = new InactiveTaxYear("0910");
    public static final InactiveTaxYear TAX_YEAR_1011 = new InactiveTaxYear("1011");
    public static final InactiveTaxYear TAX_YEAR_1112 = new InactiveTaxYear("1112");
    public static final InactiveTaxYear TAX_YEAR_1213 = new InactiveTaxYear("1213");
    public static final InactiveTaxYear TAX_YEAR_1314 = new InactiveTaxYear("1314");
    public static final InactiveTaxYear TAX_YEAR_1415 = new InactiveTaxYear("1415");
    public static final InactiveTaxYear TAX_YEAR_1516 = new InactiveTaxYear("1516");
    public static final InactiveTaxYear TAX_YEAR_1617 = new InactiveTaxYear("1617");
    public static final InactiveTaxYear TAX_YEAR_1718 = new InactiveTaxYear("1718");
    public static final InactiveTaxYear TAX_YEAR_1819 = new InactiveTaxYear("1819");
    public static final ActiveTaxYear TAX_YEAR_1920 = new ActiveTaxYear("1920");
    public static final ActiveTaxYear TAX_YEAR_2021 = new ActiveTaxYear("2021");
    public static final ActiveTaxYear TAX_YEAR_2122 = new ActiveTaxYear("2122");
    public static final ActiveTaxYear TAX_YEAR_2223 = new ActiveTaxYear("2223");
    public static final ActiveTaxYear TAX_YEAR_2324 = new ActiveTaxYear("2324");

    public static final TaxYear UNSUPPORTED = new InactiveTaxYear("Tax Year Unsupported");

    public static final TaxYear UNKNOWN = new InactiveTaxYear("Tax Year Unknown");

    private static final ActiveTaxYear[] ALL_ACTIVE = new ActiveTaxYear[]{
            TAX_YEAR_1920, TAX_YEAR_2021, TAX_YEAR_2122, TAX_YEAR_2223, TAX_YEAR_2324
    };

    private static final TaxYear[] ALL = new TaxYear[]{TAX_YEAR_0304, TAX_YEAR_0405, TAX_YEAR_0506,
            TAX_YEAR_0607, TAX_YEAR_0708, TAX_YEAR_0809, TAX_YEAR_0910, TAX_YEAR_1011,
            TAX_YEAR_1112, TAX_YEAR_1213, TAX_YEAR_1314, TAX_YEAR_1415, TAX_YEAR_1516,
            TAX_YEAR_1617, TAX_YEAR_1718, TAX_YEAR_1819, TAX_YEAR_1920, TAX_YEAR_2021,
            TAX_YEAR_2122, TAX_YEAR_2223, TAX_YEAR_2324};

    private static final ActiveTaxYear[] NONE = new ActiveTaxYear[0];

    public static ActiveTaxYear[] allActive() {
        return ALL_ACTIVE;
    }

    public static ActiveTaxYear[] since(ActiveTaxYear startingYear) {
        int startingPos = positionOf(startingYear);
        if (startingPos >= 0) {
            ActiveTaxYear[] result = new ActiveTaxYear[ALL_ACTIVE.length - startingPos];
            for (int i = startingPos, j = 0; i < ALL_ACTIVE.length; i++, j++) {
                result[j] = ALL_ACTIVE[i];
            }
            return result;
        }
        return NONE;
    }

    public static ActiveTaxYear[] only(ActiveTaxYear year) {
        return new ActiveTaxYear[]{year};
    }

    public static ActiveTaxYear[] between(ActiveTaxYear startingYear, ActiveTaxYear endingYear) {
        int startingPos = positionOf(startingYear);
        int endingPos = positionOf(endingYear);

        assert startingPos < endingPos : "Starting year must be earlier than ending year";

        if (startingPos >= 0) {
            ActiveTaxYear[] result = new ActiveTaxYear[endingPos - startingPos + 1];
            for (int i = startingPos, j = 0; i <= endingPos; i++, j++) {
                result[j] = ALL_ACTIVE[i];
            }
            return result;
        }
        return NONE;
    }

    public static ActiveTaxYear[] until(ActiveTaxYear startingYear) {
        int startingPos = positionOf(startingYear);
        if (startingPos >= 0) {
            ActiveTaxYear[] result = new ActiveTaxYear[startingPos + 1];
            for (int i = 0; i <= startingPos; i++) {
                result[i] = ALL_ACTIVE[i];
            }
            return result;
        }
        return NONE;
    }

    private static int positionOf(ActiveTaxYear startingYear) {
        int startingPos = -1;
        for (int i = 0; i < ALL_ACTIVE.length; i++) {
            if (ALL_ACTIVE[i] == startingYear) {
                startingPos = i;
                break;
            }
        }
        return startingPos;
    }

    private static final ThreadLocal<IRNamespaceMapping> IR_NS_MAPPING = new ThreadLocal<IRNamespaceMapping>();

    public static void setIRNamespaceMapping(final IRNamespaceMapping mapping) {
        IR_NS_MAPPING.set(mapping);
    }

    static {
        addValue(TAX_YEAR_0304);
        addValue(TAX_YEAR_0405);
        addValue(TAX_YEAR_0506);
        addValue(TAX_YEAR_0607);
        addValue(TAX_YEAR_0708);
        addValue(TAX_YEAR_0809);
        addValue(TAX_YEAR_0910);
        addValue(TAX_YEAR_1011);
        addValue(TAX_YEAR_1112);
        addValue(TAX_YEAR_1213);
        addValue(TAX_YEAR_1314);
        addValue(TAX_YEAR_1415);
        addValue(TAX_YEAR_1516);
        addValue(TAX_YEAR_1617);
        addValue(TAX_YEAR_1718);
        addValue(TAX_YEAR_1819);
        addValue(TAX_YEAR_1920);
        addValue(TAX_YEAR_2021);
        addValue(TAX_YEAR_2122);
        addValue(TAX_YEAR_2223);
        addValue(TAX_YEAR_2324);
    }

    private static void addValue(final TaxYear taxYear) {
        taxYearMap.put(taxYear.toString(), taxYear);
    }

    protected TaxYear(final String yearString) {
        this.yearString = yearString;
    }

    @Override
    public String toString() {
        return yearString;
    }

    public static TaxYear fromRawString(final String rawTaxYearString) {
        return taxYearMap.get(rawTaxYearString);
    }

    public static TaxYear fromString(final String taxYearString) {
        final TaxYear result = fromRawString(taxYearString);
        if (result == null) {
            throw new IllegalArgumentException("Cannot map '" + taxYearString + "' to a TaxYear");
        }
        return result;
    }

    public static TaxYear estimateFromCurrentDate() {
        final Calendar cal = Calendar.getInstance();
        cal.set(Calendar.MONTH, Calendar.APRIL);
        cal.set(Calendar.DATE, 15);
        cal.clear(Calendar.HOUR);
        cal.clear(Calendar.MINUTE);
        cal.clear(Calendar.SECOND);
        cal.clear(Calendar.MILLISECOND);
        return estimateFromPeriodEndDate(cal.getTime());
    }

    /**
     * Estimates the tax year given a period end date. Note that this is an
     * estimate only, used in order to bootstrap the validation process. A given
     * PeriodEndRule implementation actually performs the true period end
     * validation using the correct calculation.
     */
    public static TaxYear estimateFromPeriodEndDate(final Date periodEndDate) {
        final TaxYear result;
        if (!periodEndDate.before(DateHelper.april6th2003) && periodEndDate.before(DateHelper.april6th2004)) {
            result = TAX_YEAR_0304;
        } else if (!periodEndDate.before(DateHelper.april6th2004) && periodEndDate.before(DateHelper.april6th2005)) {
            result = TAX_YEAR_0405;
        } else if (!periodEndDate.before(DateHelper.april6th2005) && periodEndDate.before(DateHelper.april6th2006)) {
            result = TAX_YEAR_0506;
        } else if (!periodEndDate.before(DateHelper.april6th2006) && periodEndDate.before(DateHelper.april6th2007)) {
            result = TAX_YEAR_0607;
        } else if (!periodEndDate.before(DateHelper.april6th2007) && periodEndDate.before(DateHelper.april6th2008)) {
            result = TAX_YEAR_0708;
        } else if (!periodEndDate.before(DateHelper.april6th2008) && periodEndDate.before(DateHelper.april6th2009)) {
            result = TAX_YEAR_0809;
        } else if (!periodEndDate.before(DateHelper.april6th2009) && periodEndDate.before(DateHelper.april6th2010)) {
            result = TAX_YEAR_0910;
        } else if (!periodEndDate.before(DateHelper.april6th2010) && periodEndDate.before(DateHelper.april6th2011)) {
            result = TAX_YEAR_1011;
        } else if (!periodEndDate.before(DateHelper.april6th2011) && periodEndDate.before(DateHelper.april6th2012)) {
            result = TAX_YEAR_1112;
        } else if (!periodEndDate.before(DateHelper.april6th2012) && periodEndDate.before(DateHelper.april6th2013)) {
            result = TAX_YEAR_1213;
        } else if (!periodEndDate.before(DateHelper.april6th2013) && periodEndDate.before(DateHelper.april6th2014)) {
            result = TAX_YEAR_1314;
        } else if (!periodEndDate.before(DateHelper.april6th2014) && periodEndDate.before(DateHelper.april6th2015)) {
            result = TAX_YEAR_1415;
        } else if (!periodEndDate.before(DateHelper.april6th2015) && periodEndDate.before(DateHelper.april6th2016)) {
            result = TAX_YEAR_1516;
        } else if (!periodEndDate.before(DateHelper.april6th2016) && periodEndDate.before(DateHelper.april6th2017)) {
            result = TAX_YEAR_1617;
        } else if (!periodEndDate.before(DateHelper.april6th2017) && periodEndDate.before(DateHelper.april6th2018)) {
            result = TAX_YEAR_1718;
        } else if (!periodEndDate.before(DateHelper.april6th2018) && periodEndDate.before(DateHelper.april6th2019)) {
            result = TAX_YEAR_1819;
        } else if (!periodEndDate.before(DateHelper.april6th2019) && periodEndDate.before(DateHelper.april6th2020)) {
            result = TAX_YEAR_1920;
        } else if (!periodEndDate.before(DateHelper.april6th2020) && periodEndDate.before(DateHelper.april6th2021)) {
            result = TAX_YEAR_2021;
        } else if (!periodEndDate.before(DateHelper.april6th2021) && periodEndDate.before(DateHelper.april6th2022)) {
            result = TAX_YEAR_2122;
        } else if (!periodEndDate.before(DateHelper.april6th2022) && periodEndDate.before(DateHelper.april6th2023)) {
            result = TAX_YEAR_2223;
        } else if (!periodEndDate.before(DateHelper.april6th2023) && periodEndDate.before(DateHelper.april6th2024)) {
            result = TAX_YEAR_2324;
        } else {
            result = UNSUPPORTED;
        }
        return result;
    }

    public String getIREnvelopeNamespaceString() {
        return IR_NS_MAPPING.get().getIRNamespaceForTaxYear(this);
    }

    public static TaxYear[] getAllTaxYears() {
        final TaxYear[] result = new TaxYear[taxYearMap.size()];
        int i = 0;
        for (final String key : taxYearMap.keySet()) {
            result[i++] = taxYearMap.get(key);
        }
        return result;
    }

    public static String[] getUniqueNamespaces() {
        final TaxYear[] taxYears = getAllTaxYears();
        final Set<String> namespaces = new HashSet<String>();
        for (final TaxYear element : taxYears) {
            final String namespace = element.getIREnvelopeNamespaceString();
            if (namespace != null) {
                namespaces.add(namespace);
            }
        }

        final String[] namespaceStrings = new String[namespaces.size()];
        int namespaceCounter = 0;
        for (final String namespaceString : namespaces) {
            namespaceStrings[namespaceCounter] = namespaceString;
            namespaceCounter++;
        }
        return namespaceStrings;
    }

    public static String[] getNamespacesSince(ActiveTaxYear firstYear) {
        final TaxYear[] taxYears = TaxYear.since(firstYear);
        final String[] namespaceStrings = new String[taxYears.length];
        for (int i = 0; i < taxYears.length; i++) {
            namespaceStrings[i] = taxYears[i].getIREnvelopeNamespaceString();
        }
        return namespaceStrings;
    }

    public abstract boolean isActive();

    public boolean isAfter(TaxYear forComparison) {
        boolean result = false;
        if (this != forComparison) {
            for (TaxYear ty : ALL) {
                if (ty == forComparison) {
                    result = true;
                    break;
                } else if (ty == this) {
                    break;
                }
            }
        }
        return result;
    }

    public boolean isBefore(TaxYear forComparison) {
        boolean result = false;
        if (this != forComparison) {
            for (TaxYear ty : ALL) {
                if (ty == this) {
                    result = true;
                    break;
                } else if (ty == forComparison) {
                    break;
                }
            }
        }
        return result;
    }

    public boolean isSince(ActiveTaxYear forComparison) {
        boolean result = false;
        for (TaxYear ty : since(forComparison)) {
            if (ty == this) {
                result = true;
                break;
            }
        }
        return result;
    }
}
