/*
 * Created on 12-Dec-2005
 * 
 * $Id: ToleranceValidatorImpl.java,v 1.8 2006/08/10 15:19:13 dec Exp $
 */
package paye.tolerances.validator;

import hmrc.bizrules.QualityStage;

import java.util.List;

import paye.tolerances.Tolerable;
import paye.tolerances.Tolerance;
import paye.tolerances.ToleranceResult;
import paye.tolerances.ToleranceResultBean;
import paye.tolerances.parameters.ToleranceParameters;
import ericResponse.output.error.EricResponseError;

/**
 * Provides an implementation of the tolerance validation checks for a PAYE EoY
 * submission.
 * 
 * $Id: ToleranceValidatorImpl.java,v 1.8 2006/08/10 15:19:13 dec Exp $
 */
class ToleranceValidatorImpl implements ToleranceValidator {
    private final ToleranceResultBean toleranceResult = new ToleranceResultBean();

    private final List<EricResponseError> ericResponseErrors;

    /**
     * Create an instance of the tolerance validator with given parameters and
     * details of the submission under test to allow the checks to be carried
     * out.
     * 
     * @param toleranceParameters
     *            the tolerance levels which are to be applied
     * @param ericResponseErrors
     *            the list of errors from the submission validation
     * @param p14Count
     *            the number of P14s which in the submission being checked. Used
     *            to check the QS2 error threshold
     * @param totalPayable
     *            the total amount of tax payable. Used to check the cash
     *            tolerance threshold
     */
    ToleranceValidatorImpl(final ToleranceParameters toleranceParameters, final List<EricResponseError> ericResponseErrors,
            final int p14Count, final double totalPayable) {
        this.ericResponseErrors = ericResponseErrors;
        checkTolerances();
    }

    /**
     * @return a ToleranceResult which contains the result of the tolerance
     *         checks for both QS2 and QS3.
     */
    public ToleranceResult getToleranceResult() {
        return toleranceResult;
    }

    private void checkTolerances() {
        for (final Object error : ericResponseErrors) {
            Tolerable tolerable = null;
            Tolerance tolerance = Tolerance.TYPE_1;

            if (error instanceof Tolerable) {
                tolerable = (Tolerable) error;
                tolerance = tolerable.getTolerance();
            }

            if (tolerance == Tolerance.TYPE_1) {
                /*
                 * Any Type 1 rule violation will cause the submission to fail
                 * validation.
                 * 
                 * NOTE: tolerable == null means the error instance was probably
                 * generated by some generic process, since it does not
                 * implement the Tolerable interface, in this case the stage can
                 * always be considered to be QS2
                 */
                if (tolerable == null || tolerable.getStage() == QualityStage.QS2) {
                    toleranceResult.failedQS2Tolerance();
                } else {
                    toleranceResult.failedQS3Tolerance();
                }
            } else if (tolerance == Tolerance.DNA) {
                /*
                 * DNA errors are simply ignored. You can have as many as you
                 * like and the submission will not fail.
                 */
            }
        }
    }

}
