package processingError;

import java.util.List;

/**
 * This declares the interface for all the different kinds of errors which can
 * be generated by the processing of a document.
 * <p>
 * Known subclasses are DocumentError, BusinessRuleError and DecompressionError.
 * 
 * @author Doug Clinton
 */
public interface ProcessingError {
    /**
     * A useful property name for storing the extraInfo object in a map, or
     * elsewhere.
     */
    public static final String ERROR_EXTRA_INFO = "ERROR_EXTRA_INFO";

    /**
     * The rule name identifies the particular rule which was violated. The
     * implementing class needs to provide a meaningful value for this. For
     * example, BusinessRule would provide the name of the business rule, but
     * DocumentError might provide the element XPath to identify the particular
     * XML or Schema rule which was violated.
     * 
     */
    String getRuleName();

    /**
     * The subCode provides a further distinction about which part of the rule
     * was violated.
     */
    String getSubCode();

    /**
     * This object provides more detail about the value which caused the error.
     * It might be a simple string or some more complex object. This is a
     * placeholder so that the Object can be carried through to the particular
     * output stage and be interpreted at that point. It can be <b>null</b>.
     */
    Object getValueInError();

    /**
     * This tells us where in the XML the error was detected. It can be <b>null</b>
     */
    Location getLocation();

    /**
     * Returns the location in XPath format (e.g. /a[1]/b[3]/c[2]). Might be
     * null.
     */
    String getXPathLocation();

    /**
     * 
     * @return a text description of the error. Useful for diagnostics.
     */
    String getDescription();

    /**
     * This is a placeholder for any other information which the creator of the
     * error thinks might be useful at some other point in processing. It is up
     * to the various processing stages to coordinate on the meaning of this for
     * a particular error.
     */
    Object getExtraInfo();

    /**
     * ProcessingErrors can get chained as they move through the various
     * processing stages and get transformed. A DocumentError can be transformed
     * to a BusinessRuleError, for example. At each transformation we chain the
     * earlier error onto the new error so that we have a history of how things
     * progressed.
     * 
     * @return
     */
    ProcessingError getChainedError();

    /**
     * At the end of the chain of errors there will be an original error that
     * kicked things off. This method will return that error. Because it is
     * possible that the original error was not a ProcessingError (say because
     * it came from a SAX parser), it is returned as an Object and it is up to
     * the caller to interpret it.
     * 
     * @return the error at the start of the chain. If this error is the start
     *         of the chain then it will, itself, be returned. This means that
     *         this method should never return null.
     */
    Object getOriginalError();

    /**
     * Returns the ErrorProcessor that was used to create this error. This
     * allows traceability of error creation through the system.
     */
    ErrorProcessor getErrorProcessor();

    String getErrorCode();

    String getMessage();

    /**
     * Return a list of data items that the application may have attached to
     * this error.
     * 
     * @return List, possibly empty, of data Objects that the application has
     *         attached to this error.
     */
    List<?> getApplicationData();
}
