/*
 * Created on Feb 3, 2006
 * 
 * Purpose
 * This program will read some template xml files and selectively remove mandatory tags.
 * It will also generate empty tags ("<Tag></Tag>") and self-closing tags ("<Tag/>")
 * 
 */
package subproc.tools;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

/**
 * @author Gary
 * 
 */
public class MandatoryFieldCreator {

    /**
     * There will be no null fields test cases generated for these elements ( as
     * they are not mandatory fields )
     */
    private static String[] ignoreList = new String[] { "<IRheader>", "<Keys>", "<Principal>", "<Contact>", "<Name>",
            "<Telephone>", "<EndOfYearReturn>", "<P35>", "<EmployerDetails>", "<CheckList>", "<Declarations>", "<Summary>",
            "<NIC>", "<PAYE>", "<CombinedAmounts>", "<P14>", "<Name>", "<Address>", "<NICs", "<Earn>", "<PayAndTax>", "<PrevEmp>",
            "<CompressedPart>", "<Country>", "<ECON>", "<NINO>", "<DOB>", "<PostCode>", "<SCON>", "<Start>", "<EndDate>", "<Code>",
            "<Week53Indicator>", "<P14Count>" };

    private static boolean tagHasAnAttributeWithinIt(String tag) {
        tag = tag.trim();

        // Find start tag
        final int startTagIndex = tag.indexOf("<");
        if (startTagIndex == -1) {
            return false;
        }

        // Find end tag
        final int endTagIndex = tag.indexOf(">", startTagIndex);
        if (endTagIndex == -1) {
            return false;
        }

        // Find a space within tag
        final int spaceIndex = tag.indexOf(" ", startTagIndex);
        if (spaceIndex < endTagIndex && spaceIndex != -1) {
            return true;
        }
        return false;
    }

    private static boolean tagIsIgnorable(final String tag) {
        for (final String element : ignoreList) {
            if (tag.indexOf(element) != -1) {
                return true;
            }
        }
        return false;

    }

    private static void stripAndGenerate(final StringBuffer xml, final String destPath, final String returnType,
            final String submissionType) {
        int nextPos = xml.indexOf("<IRheader>");
        String tag;

        while ((nextPos = xml.indexOf("\n", nextPos + 1)) != -1) {
            try {
                tag = xml.substring(nextPos + 1, xml.indexOf("\n", nextPos + 2)).trim();
            } catch (final StringIndexOutOfBoundsException e) {
                System.out.println("END OF DOCUMENT >>" + xml.substring(nextPos + 1));
                return;
            }

            final int startPos = tag.indexOf("<");
            if (startPos > -1)
                if (tag.substring(startPos + 1, startPos + 2).equals("/")) {
                    // This is an end-tag.. ignore it.
                    continue;
                }
            // See if we need to ignore this tag
            if (tagIsIgnorable(tag)) {
                continue;
            }

            // Don't need to ignore tag... take a copy of string buffer and
            // create three separate files:-
            //
            // 1. Create a <tag/>
            // 2. Create <tag><tag/>
            // 3. Remove what's there currently

            String oneTag = null;
            String emptyTag = null;

            if (tagHasAnAttributeWithinIt(tag)) {
                System.out.println("Ignoring" + tag);
                continue;
            }

            final String tagName = tag.substring(1, tag.indexOf(">"));

            // Generate the a <tagname/>
            oneTag = "<" + tagName + "/>";

            // Generate an tag with no element eg <tagname></tagname>
            emptyTag = "<" + tagName + ">" + "</" + tagName + ">";

            // Scan the XML, replace the current line with a newly-generated
            // one
            final int replaceFrom = xml.indexOf(tag);
            final int replaceTo = xml.indexOf("\n", replaceFrom);

            {
                // Replace current tag with "<tagname/>"
                final StringBuffer newXml = new StringBuffer(xml.toString());
                replace(newXml, replaceFrom, replaceTo, oneTag);
                writeFile(newXml, destPath, tagName + "-empty", returnType, submissionType);
            }

            {
                // Replace current tag with "<tagname></tagname>"
                final StringBuffer newXml = new StringBuffer(xml.toString());
                replace(newXml, replaceFrom, replaceTo, emptyTag);
                writeFile(newXml, destPath, tagName + "-blank", returnType, submissionType);
            }

            {
                // Delete current tag
                final StringBuffer newXml = new StringBuffer(xml.toString());
                replace(newXml, replaceFrom, replaceTo, "<!-- removed " + tagName + "-->");
                writeFile(newXml, destPath, tagName + "-removed", returnType, submissionType);
            }
        }
    }

    private static void replace(final StringBuffer xml, final int replaceFrom, final int replaceTo, final String replaceWith) {
        xml.replace(replaceFrom, replaceTo, replaceWith);
    }

    private static void writeFile(final StringBuffer xml, final String destPath, final String tagName, final String returnType,
            final String subType) {
        try {
            final String fileName = destPath + File.separator + "error-mandatory-" + returnType + "-" + subType + "-" + tagName
                    + ".xml";
            final Writer writer = new FileWriter(fileName);

            writer.write(xml.toString());
            writer.close();
        } catch (final IOException e1) {
            e1.printStackTrace();
        }
    }

    public static StringBuffer readFileIntoStringBuffer(final String fileName) {

        final StringBuffer sb = new StringBuffer();
        String line = null;
        FileReader fr = null;
        BufferedReader br = null;

        try {
            fr = new FileReader(fileName);
            br = new BufferedReader(fr);

            while ((line = br.readLine()) != null) {
                sb.append(line + "\n");
            }

        } catch (final FileNotFoundException e) {
            e.printStackTrace();

        } catch (final IOException e) {
            e.printStackTrace();

        } finally {
            try {
                if (br != null)
                    br.close();
            } catch (final IOException ioe) {
                // stifle
            }

            try {
                if (fr != null)
                    fr.close();
            } catch (final IOException ioe) {
                // stifle

            }
        }

        return sb;
    }

    public static void main(final String[] args) {

        final long startTime = System.currentTimeMillis();

        final String pathName = "C:/temp/";

        stripAndGenerate(readFileIntoStringBuffer(pathName + "0506-test-orig-complete.xml"), pathName, "original", "complete");
        stripAndGenerate(readFileIntoStringBuffer(pathName + "0506-test-amended-complete.xml"), pathName, "amended", "complete");

        stripAndGenerate(readFileIntoStringBuffer(pathName + "0506-test-orig-P14Part.xml"), pathName, "original", "P14Part");
        stripAndGenerate(readFileIntoStringBuffer(pathName + "0506-test-amended-P14Part.xml"), pathName, "amended", "P14Part");

        stripAndGenerate(readFileIntoStringBuffer(pathName + "0506-test-orig-P35Part.xml"), pathName, "original", "P35Part");
        stripAndGenerate(readFileIntoStringBuffer(pathName + "0506-test-amended-P35Part.xml"), pathName, "amended", "P35Part");

        final long endTime = System.currentTimeMillis();

        System.out.println("Complete in " + (endTime - startTime) + "ms");
    }
}
