package xmldoc;

import com.gsl.util.collections.NestedHashMapTwoKeyMap;
import com.gsl.util.collections.TwoKeyMap;

/**
 * A SimpleElementReference brings together strings for a namespace and an XPath
 * into a bean which implements the ElementReference interface.
 * 
 * <br>
 * You can only instantiate a SimpleElementReference using the static
 * <em>makeReference()</em> method. This ensures that making two references
 * with the same namespace and XPath will always return the same java object.
 * For this reason we do not provide an <em>equals</em> method since the
 * standard java semantics of using object identity will work perfectly well.
 * 
 * This also helps performance by eliminating a vast amount of garbage
 * collection - SimpleElementReferences are used extensively throughout the
 * system and without the caching many hundreds would be created even for each
 * small submission.
 * 
 * @author Douglas Clinton
 * @since Jan 5, 2006
 * 
 */
public class SimpleElementReference implements ElementReference {

    private static final long serialVersionUID = -6477236450321651159L;

    private static final TwoKeyMap<String, String, ElementReference> referenceCache = new NestedHashMapTwoKeyMap<String, String, ElementReference>();

    private final String xPath;

    private final String namespaceURL;

    public static ElementReference makeReference(final String xPath) {
        return makeReference(null, xPath);
    }

    /**
     * @return an ElementReference combining the <em>namespsaceURL</em> and
     *         the <em>xPath</em>. The same combination of
     *         <em>namespsaceURL</em> and <em>xPath</em> will always return
     *         the same java object so it is perfectly fine to use "==" instead
     *         of <em>equals()</em> to compare two SimpleElementReferences.
     */
    public static ElementReference makeReference(final String namespaceURL, final String xPath) {
        ElementReference ref = (ElementReference) referenceCache.get(namespaceURL, xPath);
        if (ref == null) {
            synchronized (referenceCache) {
                ref = (ElementReference) referenceCache.get(namespaceURL, xPath);
                if (ref == null) {
                    ref = new SimpleElementReference(namespaceURL, xPath);
                    referenceCache.put(namespaceURL, xPath, ref);
                }
            }
        }
        return ref;
    }

    /**
     * This is private so that the only way to create a SimpleElementReference
     * is via the <em>makeReference</em> method.
     */
    private SimpleElementReference(final String namespaceurl, final String path) {
        namespaceURL = namespaceurl;
        xPath = path;
    }

    public String getNamespaceURL() {
        return namespaceURL;
    }

    public String getXPath() {
        return xPath;
    }

    @Override
    public String toString() {
        return "SimpleElementReference [xpath='" + xPath + "', namespace='" + namespaceURL + "']";
    }
}
