package xmldoc.sax;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.xml.sax.Attributes;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

import processingError.ErrorLimitExceeded;
import processingError.ProcessingErrorCollector;
import processingError.SimpleErrorCollector;
import xmldoc.Attribute;
import xmldoc.AttributeDetail;
import xmldoc.DocumentError;
import xmldoc.DocumentEventReceiver;
import xmldoc.DocumentLocation;
import xmldoc.ElementReference;
import xmldoc.SimpleElementReference;

public abstract class AbstractDocumentHandler extends DefaultHandler implements DocumentHandler {

    /**
     * This is the object to which the enriched events will be dispatched.
     */
    protected final DocumentEventReceiver docEventReceiver;

    protected ProcessingErrorCollector errorCollector;

    protected EntityResolver entityResolver;

    protected SAXExceptionDecoder exceptionDecoder = new NullSAXExceptionDecoder();

    protected ElementReference currentElementRef = SimpleElementReference.makeReference("", "");

    private boolean dumpBusinessErrorsOnDocumentError = false;

    private final List<DocumentErrorFilter> documentErrorFilters = new ArrayList<DocumentErrorFilter>();

    private boolean interrupted = false;

    public AbstractDocumentHandler(final DocumentEventReceiver docEventReceiver) {
        this.docEventReceiver = docEventReceiver;
    }

    public DocumentEventReceiver getDocEventReceiver() {
        return docEventReceiver;
    }

    public void interrupt() {
        interrupted = true;
    }

    protected final void checkTerminated() {
        if (interrupted) {
            throw new TerminateParse();
        }
    }

    @Override
    public void startDocument() throws SAXException {
        checkTerminated();

        // this call initialises error collector if it's null at this point
        getErrorCollector();

        docEventReceiver.startDocument();
    }

    protected Attribute[] createAttributeArray(final Attributes attributes) {
        final Attribute[] attrs = new Attribute[attributes.getLength()];
        for (int i = 0; i < attrs.length; i++) {
            attrs[i] = new AttributeDetail(attributes.getLocalName(i), attributes.getValue(i));
        }
        return attrs;
    }

    @Override
    public void characters(final char[] ch, final int start, final int length) throws SAXException {
        checkTerminated();

        docEventReceiver.elementText(currentElementRef, ch, start, length);
    }

    @Override
    public void endDocument() throws SAXException {
        checkTerminated();

        docEventReceiver.endDocument();
    }

    protected DocumentError createDocumentError(final SAXParseException ex) {
        final DocumentError error = exceptionDecoder.decodeException(currentElementRef, ex);
        error.setLocation(new DocumentLocation(ex, error.getElementReference()));

        final Object extraInfo = getErrorCollector().getErrorExtraInfo();
        error.setExtraInfo(extraInfo);

        return error;
    }

    /**
     * Deal with an error by creating a DocumentError to wrap the
     * SAXParseException.
     */
    @Override
    public void error(final SAXParseException ex) throws SAXException {
        final DocumentError error = createDocumentError(ex);

        boolean shouldCollectError = true;
        for (final DocumentErrorFilter filter : documentErrorFilters) {
            if (!filter.collectError(error)) {
                shouldCollectError = false;
                break;
            }
        }

        if (shouldCollectError) {
            collectError(error);
        }
    }

    private void collectError(final DocumentError error) throws ErrorLimitExceeded {
        if (!getDocEventReceiver().handleError(error)) {
            if (!error.getSubCode().equals(DocumentError.SUPPRESS)) {

                /*
                 * Dump any non-document errors prior to trying to add the
                 * current error. This ensures that the error being added won't
                 * trigger a 'too many errors' exception if there are some
                 * existing errors which we can dump.
                 * 
                 * XXX This flag is currently only used by the schematron
                 * validator and is independent of the flag which stops rules
                 * being processed following a document error.
                 */
                if (getDumpBusinessErrorsOnDocumentError() && !getErrorCollector().getErrors().isEmpty()) {
                    getErrorCollector().dumpErrorsNotOfType(DocumentError.class);
                }
                getErrorCollector().addError(error);
            }
        }
    }

    /**
     * Warnings can be handled just the same way as non-fatal errors
     */
    @Override
    public void warning(final SAXParseException ex) throws SAXException {
        error(ex);
    }

    /**
     * Fatal errors should always be collected.
     */
    @Override
    public void fatalError(final SAXParseException ex) throws SAXException {
        final DocumentError error = createDocumentError(ex);
        collectError(error);
    }

    @Override
    public InputSource resolveEntity(final String publicId, final String systemId) throws SAXException {
        InputSource result = null;
        try {
            if (entityResolver != null) {
                result = entityResolver.resolveEntity(publicId, systemId);
            }
        } catch (final IOException e) {
            /*
             * Some strangeness here. The docs on DefaultHandler are
             * inconsistent as the method sig says resolveEntity only throws
             * SAXException but the comments say it also throws IOException.
             */
            throw new SAXException(e);
        }
        return result;
    }

    public ProcessingErrorCollector getErrorCollector() {
        if (errorCollector == null) {
            errorCollector = new SimpleErrorCollector();
        }
        return errorCollector;
    }

    public void setErrorCollector(final ProcessingErrorCollector errorCollector) {
        this.errorCollector = errorCollector;
    }

    public EntityResolver getEntityResolver() {
        return entityResolver;
    }

    public void setEntityResolver(final EntityResolver entityResolver) {
        this.entityResolver = entityResolver;
    }

    public void setExceptionDecoder(final SAXExceptionDecoder exceptionDecoder) {
        this.exceptionDecoder = exceptionDecoder;
    }

    public boolean getDumpBusinessErrorsOnDocumentError() {
        return dumpBusinessErrorsOnDocumentError;
    }

    public void setDumpBusinessErrorsOnDocumentError(final boolean dumpBusinessErrorsOnDocumentError) {
        this.dumpBusinessErrorsOnDocumentError = dumpBusinessErrorsOnDocumentError;
    }

    public void addDocumentErrorFilter(final DocumentErrorFilter filter) {
        documentErrorFilters.add(filter);
    }
}
