package xmldoc.sax;

import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import xmldoc.Attribute;
import xmldoc.DocumentEventReceiver;
import xmldoc.NamespacePrefixMapper;
import xmldoc.SimpleElementReference;

/**
 * This DocumentHandler receives all the usual SAX callbacks (startElement,
 * endElement, etc) and dispatches appropriate events off to the
 * DocumentEventReceiver. The value this adds is that the Document Events carry
 * a lot more context. Rather than just the element name for a start event, for
 * instance, the DocumentEventReceiver gets the full ElementReference for the
 * event. It is also namespace-aware.
 * 
 * @author Doug Clinton
 */
public class DocumentHandlerImpl extends AbstractDocumentHandler {

    private final Map<String, Stack<String>> currentElementStacks = new HashMap<String, Stack<String>>();

    public DocumentHandlerImpl(final DocumentEventReceiver docEventReceiver) {
        super(docEventReceiver);
    }

    @Override
    public void startElement(final String namespaceURI, final String localName, final String qName, final Attributes attributes)
            throws SAXException {
        checkTerminated();

        String currentElement = getCurrentElement(namespaceURI);
        if ("".equals(currentElement)) {
            docEventReceiver.startNamespace(namespaceURI);
        }

        getCurrentElementStack(namespaceURI).push(currentElement + "/" + localName);
        currentElement = getCurrentElement(namespaceURI);

        currentElementRef = SimpleElementReference.makeReference(namespaceURI, currentElement);
        final Attribute[] attrs = createAttributeArray(attributes);
        docEventReceiver.startElement(currentElementRef, attrs);
    }

    @Override
    public void endElement(final String namespaceURI, final String localName, final String qName) throws SAXException {
        checkTerminated();

        docEventReceiver.endElement(currentElementRef);

        getCurrentElementStack(namespaceURI).pop();

        final String currentElement = getCurrentElement(namespaceURI);
        if ("".equals(currentElement)) {
            docEventReceiver.endNamespace(namespaceURI);
        }

        currentElementRef = SimpleElementReference.makeReference(namespaceURI, currentElement);
    }

    private String getCurrentElement(final String namespaceURI) {
        String result = "";
        final Stack<String> stack = getCurrentElementStack(namespaceURI);

        if (!stack.isEmpty()) {
            result = getCurrentElementStack(namespaceURI).peek();
        }
        return result;
    }

    private Stack<String> getCurrentElementStack(final String namespaceURI) {
        Stack<String> currentElementStack = currentElementStacks.get(namespaceURI);
        if (currentElementStack == null) {
            currentElementStack = new Stack<String>();
            currentElementStack.push("");
            currentElementStacks.put(namespaceURI, currentElementStack);
        }
        return currentElementStack;
    }

    public void setNamespacePrefixMapper(final NamespacePrefixMapper namespacePrefixMapper) {
    }
}
