package xmldoc.sax;

import java.util.ArrayList;
import java.util.List;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import xmldoc.DocumentError;
import xmldoc.DocumentEventReceiver;
import xmldoc.DocumentLocation;
import xmldoc.NamespacePrefixMapper;
import xmldoc.SimpleElementReference;

/**
 * 
 */
public class DocumentHandlerImpl2 extends AbstractDocumentHandler implements DocumentErrorPostProcessor {
    protected class Fixup {
        private final DocumentError error;

        private final FixupType type;

        public Fixup(DocumentError error, FixupType type) {
            this.error = error;
            this.type = type;
        }

        public DocumentError getError() {
            return error;
        }

        public FixupType getType() {
            return type;
        }
    }

    /*
     * This is a list of DocumentErrors which should have their location paths
     * fixed up at the next startElement() event. This is necessary because
     * Xerces often raises an error() event prior to the startElement() event
     * for the actual element which caused the error.
     */
    private final List<Fixup> fixups = new ArrayList<Fixup>();

    private final XPathLocationStack xPathLocationStack = new XPathLocationStack();

    public DocumentHandlerImpl2(final DocumentEventReceiver docEventReceiver, final boolean includePrefixesInXPaths) {
        super(docEventReceiver);
        this.xPathLocationStack.setIncludePrefixesInXPathLocatorStrings(includePrefixesInXPaths);
        this.docEventReceiver.setXPathLocator(xPathLocationStack);
    }

    @Override
    public void startElement(final String namespaceURI, final String localName, final String qName, final Attributes attributes)
            throws SAXException {
        checkTerminated();

        final String previousNamespaceURI = xPathLocationStack.getCurrentNamespace();

        xPathLocationStack.push(namespaceURI, localName);

        for (Fixup fixup : fixups) {
            fixup.error.setXPathLocation(xPathLocationStack.toXPathLocatorString());
        }
        fixups.clear();

        if (!xPathLocationStack.getCurrentNamespace().equals(previousNamespaceURI)) {
            docEventReceiver.startNamespace(namespaceURI);
        }

        currentElementRef = SimpleElementReference.makeReference(xPathLocationStack.getCurrentXPath());

        docEventReceiver.startElement(currentElementRef, createAttributeArray(attributes));
    }

    @Override
    public void endElement(final String namespaceURI, final String localName, final String qName) throws SAXException {
        checkTerminated();

        docEventReceiver.endElement(currentElementRef);

        xPathLocationStack.pop();

        if (!namespaceURI.equals(xPathLocationStack.getCurrentNamespace())) {
            docEventReceiver.endNamespace(namespaceURI);
        }

        // In some cases, a fixup wants to be cleared at an end element if we
        // havn't seen a start element since it was scheduled.
        removeFixupsOfType(FixupType.UntilEndElement);

        currentElementRef = SimpleElementReference.makeReference(xPathLocationStack.getCurrentXPath());
    }

    private void removeFixupsOfType(FixupType fixupType) {
        List<Fixup> toRemove = new ArrayList<Fixup>();
        for (Fixup fixup : fixups) {
            if (fixup.type == fixupType) {
                toRemove.add(fixup);
            }
        }
        fixups.removeAll(toRemove);
    }

    @Override
    public void characters(final char[] ch, final int start, final int length) throws SAXException {
        checkTerminated();

        docEventReceiver.elementText(currentElementRef, ch, start, length);
    }

    @Override
    public void endDocument() throws SAXException {
        checkTerminated();

        docEventReceiver.endDocument();
    }

    @Override
    protected DocumentError createDocumentError(final SAXParseException ex) {
        final DocumentError error = exceptionDecoder.decodeException(currentElementRef, ex);
        error.setLocation(new DocumentLocation(ex, error.getElementReference()));

        final Object extraInfo = getErrorCollector().getErrorExtraInfo();
        error.setExtraInfo(extraInfo);

        error.setXPathLocation(xPathLocationStack.toXPathLocatorString());
        return error;
    }

    @Override
    public void setExceptionDecoder(final SAXExceptionDecoder exceptionDecoder) {
        super.setExceptionDecoder(exceptionDecoder);
        exceptionDecoder.setDocumentErrorPostProcessor(this);
    }

    public void setNamespacePrefixMapper(final NamespacePrefixMapper namespacePrefixMapper) {
        this.xPathLocationStack.setNamespacePrefixMapper(namespacePrefixMapper);
    }

    public void scheduleErrorForProcessing(DocumentError error, FixupType type) {
        fixups.add(new Fixup(error, type));
    }

    public void scheduleErrorForProcessing(final DocumentError error) {
        fixups.add(new Fixup(error, FixupType.UntilStartElement));
    }
}
