/**
 * 
 */
package xmldoc.sax;

import java.util.EmptyStackException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;

import xmldoc.NamespacePrefixMapper;

/**
 * This class keeps track of the location in the document expressed as an XPath
 * location. The resulting location might look something like: "/A[1]/B[2]/C[1]"
 * 
 * Namespaces are supported. You can push an elementName with or without a
 * namespace. If a namespace is provided then the output format is:
 * "/ns1:A[1]/ns2:B[2]/ns3:C[1]"
 * 
 * @author doug
 * 
 */
public class XPathLocationStack implements XPathLocator, Cloneable {

    private LinkedList<XPathLocationStackFrame> locationStack = new LinkedList<XPathLocationStackFrame>();

    private NamespacePrefixMapper namespacePrefixMapper = null;

    private boolean includePrefixesInXPathLocatorStrings = true;

    @Override
    public Object clone() {
        final XPathLocationStack clone = new XPathLocationStack();
        clone.includePrefixesInXPathLocatorStrings = includePrefixesInXPathLocatorStrings;
        clone.namespacePrefixMapper = namespacePrefixMapper;
        clone.locationStack = new LinkedList<XPathLocationStackFrame>();

        final Iterator<XPathLocationStackFrame> iter = locationStack.iterator();
        while (iter.hasNext()) {
            final XPathLocationStackFrame frame = iter.next();
            clone.locationStack.add(frame.clone());
        }

        return clone;
    }

    public NamespacePrefixMapper getNamespacePrefixMapper() {
        return namespacePrefixMapper;
    }

    public void setNamespacePrefixMapper(final NamespacePrefixMapper namespacePrefixMapper) {
        this.namespacePrefixMapper = namespacePrefixMapper;
    }

    public void setIncludePrefixesInXPathLocatorStrings(final boolean includePrefixesInXPathLocatorStrings) {
        this.includePrefixesInXPathLocatorStrings = includePrefixesInXPathLocatorStrings;
    }

    public boolean isIncludePrefixesInXPathLocatorStrings() {
        return includePrefixesInXPathLocatorStrings;
    }

    private XPathLocationStackFrame getCurrentStackFrame() {
        if (locationStack.isEmpty()) {
            return null;
        }
        return locationStack.getLast();
    }

    public String getCurrentNamespace() {
        final XPathLocationStackFrame frame = getCurrentStackFrame();
        return (frame != null) ? frame.getNamespace() : null;
    }

    public String getCurrentXPath() {
        final XPathLocationStackFrame frame = getCurrentStackFrame();
        return (frame != null) ? frame.getXPath() : null;
    }

    public String toXPathLocatorString() {
        String result = "";
        for (final Iterator<XPathLocationStackFrame> iter = locationStack.iterator(); iter.hasNext();) {
            final XPathLocationStackFrame stackFrame = iter.next();
            final String elementName = includePrefixesInXPathLocatorStrings ? stackFrame.getPrefixedElementName() : stackFrame
                    .getElementName();
            result += "/" + elementName + "[" + stackFrame.getElementIndex() + "]";
        }
        return result;
    }

    @Override
    public String toString() {
        return toXPathLocatorString();
    }

    public void pop() {
        try {
            locationStack.removeLast();
        } catch (final NoSuchElementException e) {
            throw new EmptyStackException();
        }
    }

    public void push(final String elementName) {
        push(null, elementName);
    }

    public void push(final String namespace, final String elementName) {
        String prefixedElementName = elementName;
        if (namespacePrefixMapper != null) {
            final String prefix = namespacePrefixMapper.mapNamespaceToPrefix(namespace);
            prefixedElementName = (prefix != null ? (prefix + ":") : "") + elementName;
        }
        locationStack.addLast(new XPathLocationStackFrame(getCurrentStackFrame(), namespace, elementName, prefixedElementName));
    }
}
